/*	Copyright 2002 by Eric Postpischil, http://edp.org.
	See license information in index.html.
*/


#include	"StdAfx.h"
#include	"resource.h"

#include	<commdlg.h>
#include	<stdio.h>


// Assign numbers to child windows.
typedef	enum { piecesID, goalID, statusID, exhibitID }	ChildID;


/*	This is private information for a TriSolve window and its subroutines,
	except that the window creator needs to use info.status.window to
	dispatch messages for the status dialog.
*/
typedef struct {
		/*	Information about child windows is put into structures
			to keep it organized.  Although the structures differ,
			same member names refer to the same type of information.
		*/
		struct {
			HWND			window;
			TriDisplayInfo	info;
		}			exhibit;		// exhibit child information
		struct {
			HWND			window;				// handle of child window
			TriDisplayInfo	info;				// information for child window's use
			const char		*caption;			// window caption
			char			*title;				// portion of file name to use for title
			Bool			modified, readonly;	// status flags
			Shape			*shape;				// shape associated with window
			char			name[_MAX_PATH];	// file associated with window
		}			preview[2];		// pieces and goal child information
		struct {
			HWND			window;				// handle of a child
			StatusInfo		info;				// information for a child
			Pieces			*pieces;			// pieces Solver is using
			void			*problem;			// Solver problem context
		}			status;			// status child information

		Bool		editMode;		// flag for edit mode, versus solve mode
		WORD		exhibitView;	// ID of child currently being exhibited
		OptionsInfo	options;		// application options
}	TriSolveInfo;


// Try to save a value in the registry, and don't care about errors.
static void	TriSolveRegSet(const char *name, DWORD type,
						   CONST BYTE *data, DWORD length)
{
	HKEY	software = NULL, edp = NULL, trisolve = NULL;
	DWORD	disposition;


	RegOpenKeyEx(HKEY_CURRENT_USER, "Software",
		0, KEY_CREATE_SUB_KEY, &software);
	RegCreateKeyEx(software, "edp.org", 0, NULL,
		0, KEY_CREATE_SUB_KEY, NULL, &edp, &disposition);
	RegCreateKeyEx(edp, "TriSolve", 0, NULL,
		0, KEY_SET_VALUE, NULL, &trisolve, &disposition);
	RegSetValueEx(trisolve, name, 0, type, data, length);
	RegCloseKey(trisolve);
	RegCloseKey(edp);
	RegCloseKey(software);
}


/*	Delete the named registry subkey, after deleting all of its
	subkeys.  This is not needed under Windows 9x, as RegDeleteKey
	deletes all subkeys, but it is needed under Windows NT.
*/
static void	RegDeleteKeyRecursive(HKEY key, const char *name)
{
	/*	We can use a static buffer since the contents do not need to
		remain available after the recursive call.  This saves stack
		space, although it is wasteful in that the buffer is retained
		during all of program execution.
	*/
	static char	buffer[MAX_PATH+1];

	HKEY	subkey;


	// Open subkey to delete subsubkeys.
	if (ERROR_SUCCESS == RegOpenKeyEx(key, name, 0, 0, &subkey))
	{
		/*	Delete subsubkeys.  We only need to get the zeroth subkey
			each time, because we delete it, leaving something else as
			the zeroth subkey.
		*/
		while (ERROR_SUCCESS == RegEnumKey(subkey, 0, buffer, sizeof buffer))
			RegDeleteKeyRecursive(subkey, buffer);

		RegCloseKey(subkey);
	}

	// Delete the subkey.
	RegDeleteKey(key, name);
}


/*	Change the shape displayed in the exhibit window, if it currently
	corresponds to the specified child or the imperative is TRUE.
*/
static void	ChangeExhibit(TriSolveInfo *info, LONG wParam, Shape *shape)
{
	const WORD	childID	= LOWORD(wParam),
				mask	= HIWORD(wParam);


	// Ignore changing exhibit to exhibit.
	if (childID == exhibitID)
		return;

	// Are we changing to a new child?
	if (info->exhibitView != childID)
	{
		// Is the change optional (e.g., a solution update, not a user click)?
		if (! (mask & CE_IMPERATIVE))
			return;

		// If we are changing to a new child, force redraw.
		wParam |= MAKELONG(0, CE_REDRAW);

		info->exhibitView = childID;
	}

	// Use edit mode if we are in edit mode and are showing pieces or goal.
	if (info->editMode && (childID == piecesID || childID == goalID))
		wParam |= MAKELONG(0, CE_EDIT);

	SendMessage(info->exhibit.window, WM_SETSHAPE, wParam, (long) shape);
}


/*	Change the shape in TriDisplay window.  Update the exhibit window,
	but don't make it imperative if the new shape is NULL.
*/
static void	ChangeDisplay(TriSolveInfo *info, HWND window,
	TriDisplayInfo *displayInfo, Shape *shape)
{
	SendMessage(window, WM_SETSHAPE, MAKELONG(0, CE_REDRAW), (long) shape);
	ChangeExhibit(info,
		MAKELONG(GetWindowLong(window, GWL_ID),
			(shape == NULL ? 0 : CE_IMPERATIVE) | CE_REDRAW),
		shape);
}


/*	Enable or disable menu commands associated with given ID,
	to suit the current state of the associated file.  (E.g.,
	the command to close a pieces file is enabled if a pieces
	file is open.)
*/
static void	enableCommands(HWND window, TriSolveInfo *info, ChildID ID)
{
	static const struct {
			unsigned short	position, close, save, saveas;
	}	data[2] = {
		{ POS_PIECES_MENU, IDM_PIECES_CLOSE, IDM_PIECES_SAVE, IDM_PIECES_SAVEAS },
		{ POS_GOAL_MENU,   IDM_GOAL_CLOSE,   IDM_GOAL_SAVE,   IDM_GOAL_SAVEAS }
	};

    MENUITEMINFO	menuInfo = { sizeof menuInfo, MIIM_STATE };
	HMENU			menu;
	
	
	// Get the handle of the submenu to modify.
	menu = GetSubMenu(GetMenu(window), data[ID].position);

	// Iff there is a shape open, we should be able to close or save-as it.
	menuInfo.fState = info->preview[ID].shape == NULL ? MFS_GRAYED : 0;
	SetMenuItemInfo(menu, data[ID].close,  FALSE, &menuInfo);
	SetMenuItemInfo(menu, data[ID].saveas, FALSE, &menuInfo);

	// The pieces can also be rearranged.
	if (ID == piecesID)
		SetMenuItemInfo(menu, IDM_PIECES_REARRANGE, FALSE, &menuInfo);

	// We can also save the shape if it is not read-only.
	if (info->preview[ID].readonly)
		menuInfo.fState = MFS_GRAYED;
	SetMenuItemInfo(menu, data[ID].save,   FALSE, &menuInfo);
}


/*	Put a message in the status window about the numbers of triangles
	in the pieces and the goal.  This is a kludge; it might be nice
	to put the triangle counts somewhere else.
*/
static void	setMessage(TriSolveInfo *info)
{
	char	buffer[76];
	int		nP, nG;


	nP = info->preview[piecesID].shape == NULL ? 0 :
		info->preview[piecesID].shape->numPoints;
	nG = info->preview[goalID  ].shape == NULL ? 0 :
		info->preview[goalID  ].shape->numPoints;

	wsprintf(buffer, "The pieces have %d triangle%s.\nThe goal has %d triangle%s.",
		nP, nP == 1 ? "" : "s", nG, nG == 1 ? "" : "s");
	SendMessage(info->status.window, WM_SHOWMESSAGE, 0, (LPARAM) buffer);
}


/*	Set a title for a child window based on name of associated file.
	We could be fancier with this and honor the user's preference
	about displaying extensions with base file names.
*/
static void	setTitle(TriSolveInfo *info, ChildID ID)
{
	char	*p;


	// Find start of file name.
	for (p = info->preview[ID].title = info->preview[ID].name; *p; p++)
		if (*p == '\\' || *p == '/')
			info->preview[ID].title = p+1;

	SendMessage(info->preview[ID].window, WM_SETTEXT, 0,
		(long) (info->preview[ID].shape == NULL ?
			info->preview[ID].caption : info->preview[ID].title));
}


/*	Try to open a file and read a shape.
	If shouldSucceed is true, then the operation is expected
	to succeed, as when the user selected a file to open.
	If shouldSucceed is false, then the operation is
	optional, as when we try to open a default file.

	TRUE is returned if the file operation is successful.
	There may still be an error with the contents of the
	file, in which case the returned shape is NULL.

	Some processing is specific to whether the ID is goalID or piecesID.
*/
static Bool	readFile(TriSolveInfo *info, Bool shouldSucceed, ChildID ID)
{
	FILE	*file;


	file = fopen(info->preview[ID].name, "r");
	if (file == NULL)
	{
		if (shouldSucceed)
			DisplayCError("TriSolve: Error opening file");
		return FALSE;
	}

	/*	Once the file is open, we will not use the previous data anymore,
		even if some error occurs processing the new file.
	*/
	freeShape(info->preview[ID].shape);

	info->preview[ID].shape = readShape(file);
	fclose(file);
	
	// When reading pieces, give them colors by assigning IDs.
	if (ID == piecesID)
		extractPieces(info->preview[ID].shape, FALSE);

	ChangeDisplay(info, info->preview[ID].window,
		&info->preview[ID].info, info->preview[ID].shape);
	setTitle(info, ID);

	return TRUE;
}


/*	Write a shape to the named file.
	Return TRUE if successful.
*/
static Bool	writeFile(TriSolveInfo *info, const char *name, ChildID ID)
{
	const char	titleWriting[] = "TriSolve: Error writing file";

	FILE	*file;
	Bool	result;


	if (info->preview[ID].shape == NULL)
		return FALSE;

	// Open file.
	file = fopen(name, "w");
	if (file == NULL)
	{
		DisplayCError("TriSolve: Error opening file");
		return FALSE;
	}

	// Write shape to file and report error if one occurs.
	result = writeShape(file, info->preview[ID].shape);
	if (!result)
		DisplayCError(titleWriting);

	// Close file and check for error.
	if (EOF == fclose(file))
	{
		// If error did not occur previously, report one now.
		if (result)
			DisplayCError(titleWriting);
		return FALSE;
	}

	return result;
}


// Initialize an OPENFILENAME structure.
static void	initOFN(OPENFILENAME *ofn, HWND window, const char *title,
	char *filename, size_t size, DWORD extraFlags)
{
	static const char		filter[] =
		"TriSolve Shapes (*.tri)\0*.tri\0All Files (*.*)\0*.*\0";

	static const OPENFILENAME	ofnTemplate = {
		sizeof ofnTemplate,
				// size of structure
		NULL,	// owner window handle
		NULL,	// instance containing dialog box template
		filter,	// filter
		NULL,	// custom filter buffer
		0,		// size of custom filter buffer
		1,		// index of selected filter
		NULL,	// buffer to receive name
		0,		// size of buffer
		NULL,	// buffer to receive file name and extension
		0,		// size of buffer
		NULL,	// initial directory
		NULL,	// dialog box title
		OFN_NOCHANGEDIR | OFN_PATHMUSTEXIST,
				// flags
		0,		// returned offset to file name
		0,		// returned offset to extension
		"tri",	// default extension
		0,		// customer data
		NULL,	// hook procedure
		NULL	// dialog template resource name
	};

	*ofn = ofnTemplate;
	ofn->hwndOwner	 = window;
	ofn->lpstrFile	 = filename;
	ofn->nMaxFile	 = size;
	ofn->lpstrTitle	 = title;
	ofn->Flags		|= extraFlags;
}


// Call GetOpenFileName, check the result, and report an error if appropriate.
static Bool	checkedGetOpenFileName(OPENFILENAME *ofn)
{
	if (!GetOpenFileName(ofn))
	{
		// Report error if call failed for reason other than user cancel.
		if (CommDlgExtendedError() != 0)
			DisplayError("TriSolve: Executing Open dialog",
				"Common dialog box error.");
		return FALSE;
	}
	return TRUE;
}


// Call GetSaveFileName, check the result, and report an error if appropriate.
static Bool	checkedGetSaveFileName(OPENFILENAME *ofn)
{
	if (!GetSaveFileName(ofn))
	{
		// Report error if call failed for reason other than user cancel.
		if (CommDlgExtendedError() != 0)
			DisplayError("TriSolve: Executing Save As dialog",
				"Common dialog box error.");
		return FALSE;
	}
	return TRUE;
}


// Execute Save As command.  Return TRUE if successful.
static Bool	saveAs(HWND window, TriSolveInfo *info, ChildID ID)
{
	static const char	* const caption[2] = { "Save Pieces", "Save Goal" };

	OPENFILENAME	ofn;
	char			newName[_MAX_PATH];


	if (info->preview[ID].shape == NULL)
		return TRUE;

	// Ask user for name of file to write.
	strcpy(newName, info->preview[ID].name);
	initOFN(&ofn, window, caption[ID], newName, sizeof newName, OFN_OVERWRITEPROMPT);
	if (!checkedGetSaveFileName(&ofn))
		return FALSE;

	// Try to save shape.
	if (!writeFile(info, newName, ID))
		return FALSE;

	// Update the title and remember name as the last file.
	strcpy(info->preview[ID].name, newName);
	setTitle(info, ID);
	TriSolveRegSet(info->preview[ID].caption, REG_SZ, (BYTE *) info->preview[ID].name,
		1+strlen(info->preview[ID].name));

	info->preview[ID].modified = FALSE;
	info->preview[ID].readonly = FALSE;
	enableCommands(window, info, ID);
	
	return TRUE;
}


// Execute Save command.  Return TRUE if successful.
static Bool	saveFile(HWND window, TriSolveInfo *info, ChildID ID)
{
	if (info->preview[ID].shape == NULL)
		return TRUE;

	/*	If preview has a name and is not read-only, try to write it to
		the file.  Otherwise, or if the write fails, try Save As.
	*/
	if (info->preview[ID].name[0] == '\0' || info->preview[ID].readonly ||
			!writeFile(info, info->preview[ID].name, ID))
		return saveAs(window, info, ID);

	info->preview[ID].modified = FALSE;
	enableCommands(window, info, ID);
	return TRUE;
}


// Test whether the Close command can be executed successfully.
static Bool	closeFileTest(HWND window, TriSolveInfo *info, ChildID ID)
{
	// If there is no shape, closing it is a null operation.
	if (info->preview[ID].shape == NULL)
		return TRUE;

	// If shape has been modified, see what the user wants to do.
	if (info->preview[ID].modified)
	{
		MessageBeep(MB_ICONQUESTION);
		switch (MessageBox(window,
			ID == piecesID
				? "Pieces have been modified. Save them first?"
				: "Goal has been modified. Save it first?",
			"TriSolve", MB_YESNOCANCEL))
		{
			case IDYES:
				// Try to save file.  On error, fail.  On success, proceed.
				if (!saveFile(window, info, ID))
					return FALSE;
				break;
			case IDNO:
				// Okay, continue and wipe out the old data.
				break;
			case IDCANCEL:
			default:
				// We are canceling the close, so return failure.
				return FALSE;
		}
	}

	return TRUE;
}


/*	Execute the Close command.  This always closes the file --
	it should be called only if closeFileTest has been called
	previously and returned TRUE.
*/
static void	closeFile(HWND window, TriSolveInfo *info, ChildID ID)
{
	// Free data, nullify names and captions, and display nothing.
	info->preview[ID].name[0] = '\0';
	SendMessage(info->preview[ID].window, WM_SETTEXT, 0, (long) info->preview[ID].caption);
	ChangeDisplay(info, info->preview[ID].window, &info->preview[ID].info, NULL);
	freeShape(info->preview[ID].shape);
	info->preview[ID].shape = NULL;
	info->preview[ID].modified = FALSE;
	info->preview[ID].readonly = TRUE;
	enableCommands(window, info, ID);
	setMessage(info);
}


// Execute the New command.  Return TRUE iff successful.
static Bool	newFile(HWND window, TriSolveInfo *info, ChildID ID)
{
	static const char	* const caption[2] = { "Unnamed Pieces", "Unnamed Goal" };


	// If we cannot close the old file, do not open a new one.
	if (!closeFileTest(window, info, ID))
		return FALSE;
	closeFile(window, info, ID);

	// Get new shape and set name to "".
	info->preview[ID].shape = newShape();
	info->preview[ID].name[0] = '\0';

	// Display new shape.
	ChangeDisplay(info, info->preview[ID].window,
		&info->preview[ID].info, info->preview[ID].shape);
	
	/*	Set window caption.  Just in case newShape failed, check
		result and clear caption if it failed.  Otherwise, used
		"Unnamed [item]".
	*/
	SendMessage(info->preview[ID].window, WM_SETTEXT, 0,
		(long) (info->preview[ID].shape == NULL ?
			info->preview[ID].caption : caption[ID]));

	TriSolveRegSet(info->preview[ID].caption, REG_SZ, (BYTE *) "", 1);
	info->preview[ID].modified = FALSE;
	info->preview[ID].readonly = FALSE;
	enableCommands(window, info, ID);
	setMessage(info);

	return info->preview[ID].shape != NULL;
}


// Execute the Open command.  Return TRUE iff successful.
static Bool	openFile(HWND window, TriSolveInfo *info, ChildID ID)
{
	static const char	* const caption[2] = { "Open Pieces", "Open Goal" };

	char			name[_MAX_PATH];
	OPENFILENAME	ofn;


	// If we cannot close the old file, do not open a new one.
	if (!closeFileTest(window, info, ID))
		return FALSE;

	strcpy(name, info->preview[ID].name);
	initOFN(&ofn, window, caption[ID], name, sizeof name, OFN_FILEMUSTEXIST);
	if (!checkedGetOpenFileName(&ofn))
		return FALSE;

	/*	Close previous file now -- after we are past point where user
		can cancel the open.  The open could still fail, due to an
		unreadable file, leaving us with no open shape, but that may
		be okay.
	*/
	closeFile(window, info, ID);

	strcpy(info->preview[ID].name, name);
	info->preview[ID].readonly = !! (ofn.Flags & OFN_READONLY);

	/*	Try to open file.  If successful and we get a result
		from reading it, remember it as the last file read.
	*/
	if (readFile(info, TRUE, ID) && info->preview[ID].shape != NULL)
		TriSolveRegSet(info->preview[ID].caption, REG_SZ, (BYTE *) info->preview[ID].name,
			1+strlen(info->preview[ID].name));
	info->preview[ID].modified = FALSE;
	enableCommands(window, info, ID);
	setMessage(info);

	return info->preview[ID].shape != NULL;
}


// Process changes to edit/solve mode.
static void	setEditMode(HWND window, TriSolveInfo *info, Bool editMode)
{
    MENUITEMINFO	menuInfo = { sizeof menuInfo, MIIM_STATE };
	HMENU			menu;
	Bool			piecesChanged, goalChanged;
	

	if (editMode == info->editMode)
		return;

	info->editMode = editMode;

	// Get the handle of the submenu to modify.
	menu = GetSubMenu(GetMenu(window), POS_VIEW_MENU);

	// Set checks appropriately on Edit Mode and Solve Mode menu items.
	menuInfo.fState = info->editMode ? MFS_CHECKED : MFS_UNCHECKED;
	SetMenuItemInfo(menu, IDM_VIEW_EDITMODE,  FALSE, &menuInfo);

	menuInfo.fState = info->editMode ? MFS_UNCHECKED : MFS_CHECKED;
	SetMenuItemInfo(menu, IDM_VIEW_SOLVEMODE, FALSE, &menuInfo);

	/*	If leaving edit mode, compress pieces and goal.  Then color
		any gray triangles in the pieces.  If these result in changes,
		display the new pieces and/or goal.
	*/
	if (editMode)
		piecesChanged = goalChanged = FALSE;
	else
	{
		piecesChanged = compressShape(info->preview[piecesID].shape);
		goalChanged = compressShape(info->preview[goalID].shape);
		piecesChanged = extractPieces(info->preview[piecesID].shape, FALSE)
			|| piecesChanged;
		/*	Note the two piecesChanged statements cannot be combined
			as we want both function calls to be evaluated.
		*/

		if (piecesChanged)
			ChangeDisplay(info, info->preview[piecesID].window,
				&info->preview[piecesID].info, info->preview[piecesID].shape);
		if (goalChanged)
			ChangeDisplay(info, info->preview[goalID].window,
				&info->preview[goalID].info, info->preview[goalID].shape);
		/*	ChangeDisplay forces a redraw which is not necessary for
			a recoloring, but it is a single display update, so it is
			not worth bothering doing anything to reduce flicker.
		*/
	}
	if (!piecesChanged && !goalChanged)
		// Inform exhibit window of change to/from edit mode.
		ChangeExhibit(info, MAKELONG(info->exhibitView, 0),
			info->preview[info->exhibitView].shape);
}


/*	Decide where to put pieces preview and goal preview windows.
	Coordinates for CreateWindow or MoveWindow are returned in
	RECT structures.  The RECT members right and bottom actually
	contain the width and height.
*/
static void	placePreviews(RECT *pieces, RECT *goal,
	int width, int height, int widthStatus, int heightStatus)
{
	const int	hp = SMALLER_HALF(height - heightStatus);


	// If the windows are too short, put them side-by-side.
	if (3*hp < widthStatus)
	{
		SetRect(pieces, 0, heightStatus,
			SMALLER_HALF(widthStatus), height-heightStatus);
		SetRect(goal, SMALLER_HALF(widthStatus), heightStatus,
			LARGER_HALF(widthStatus), height-heightStatus);
	}
	// Usually, put the windows one above the other.
	else
	{
		SetRect(pieces, 0, heightStatus, widthStatus, hp);
		SetRect(goal, 0, heightStatus+hp, widthStatus, height-(heightStatus+hp));
	}
}


// This is the window procedure for the TriSolve class.
static LRESULT CALLBACK	TriSolveWindow(
	HWND window, UINT message, WPARAM wParam, LPARAM lParam)
{
	const char	helpTitle[] = "TriSolve: Starting Windows Help";

	static long	heightStatus, widthStatus;	// height and width of status window
	/*	(These can be static as they would not change with
		more than one TriSolve window.)
	*/

	/*	On creation, we are passed a pointer to space allocated for us.
		On subsequent calls, we use our saved copy of that pointer.
	*/
	TriSolveInfo	*const info = message == WM_CREATE
		? (TriSolveInfo *) ((LPCREATESTRUCT) lParam)->lpCreateParams
		: (TriSolveInfo *) GetWindowLong(window, GWL_USERDATA);

	HINSTANCE		hInstance = (HINSTANCE) GetWindowLong(window, GWL_HINSTANCE);


	switch (message)
	{
		case WM_CREATE:
		{
			static const char	title[] = "TriSolve: Creating main window";

			RECT	rect, pieces, goal;
			LONG	width, height;


			// Save pointer passed to us at creation for future calls.
			SetWindowLong(window, GWL_USERDATA, (long) info);

			// Initialize some data.
			memset(info, 0, sizeof *info);
#if NULL != 0
#error	"Since NULL is not zero, pointers in automatic objects must be initialized."
#endif
			info->preview[goalID].caption = "Goal";
			info->preview[piecesID].caption = "Pieces";

			// Set default options in case we do not get them from registry.
			info->options.interiorPen = INTERIORPEN;
			info->options.perimeterPenColor = PERIMETERPENCOLOR;
			info->options.perimeterPenPixels = PERIMETERPENPIXELS;

			/*	Pass ID to status dialog, create it, and report handle back to
				our creator.  (The creator of this window uses info->status.window
				to dispatch dialog messages to the dialog created here.)
			*/
			info->status.info.ID = statusID;
			info->status.window = CreateDialogParam(hInstance,
				MAKEINTRESOURCE(IDD_STATUS),
				window, Status, (LPARAM) &info->status.info);
			if (info->status.window == NULL)
			{
				DisplayWindowsError(title);
				DestroyWindow(window);
				return 0;
			}

			/*	Get absolute coordinates of window, to derive
				external width and height.
			*/
			GetWindowRect(info->status.window, &rect);
			widthStatus = rect.right - rect.left;
			heightStatus = rect.bottom - rect.top;

			/*	Position remaining windows inside using
				dialog box size as a guide.
			*/
			GetClientRect(window, &rect);
			width = rect.right - rect.left;
			height = rect.bottom - rect.top;

			placePreviews(&pieces, &goal, width, height, widthStatus, heightStatus);

			/*	Create display windows -- a big one for the main exhibit
				and two small ones to display pieces and the goal.
			*/
			info->preview[piecesID].window = CreateWindow("TriDisplay", "Pieces",
				WS_CAPTION | WS_CHILDWINDOW | WS_CLIPSIBLINGS | WS_VISIBLE,
				pieces.left, pieces.top, pieces.right, pieces.bottom,
				window, (HMENU) piecesID, hInstance, &info->preview[piecesID].info);
			if (info->preview[piecesID].window == NULL)
			{
				DisplayWindowsError(title);
				DestroyWindow(window);
				return 0;
			}

			info->preview[goalID].window = CreateWindow("TriDisplay", "Goal",
				WS_CAPTION | WS_CHILDWINDOW | WS_CLIPSIBLINGS | WS_VISIBLE,
				goal.left, goal.top, goal.right, goal.bottom,
				window, (HMENU) goalID, hInstance, &info->preview[goalID].info);
			if (info->preview[goalID].window == NULL)
			{
				DisplayWindowsError(title);
				DestroyWindow(window);
				return 0;
			}

			info->exhibit.window = CreateWindow("TriDisplay", NULL,
				WS_CHILDWINDOW | WS_CLIPSIBLINGS | WS_VISIBLE,
				widthStatus, 0, width-widthStatus, height,
				window, (HMENU) exhibitID, hInstance, &info->exhibit.info);
			if (info->exhibit.window == NULL)
			{
				DisplayWindowsError(title);
				DestroyWindow(window);
				return 0;
			}

			// Open our registry key and check for values.
			{
				HKEY		software = NULL, edp = NULL, trisolve = NULL;
				OptionsInfo	newOptions;
				DWORD		length, type;
				LONG		status;


				// Open TriSolve registry key.
				RegOpenKeyEx(HKEY_CURRENT_USER, "Software", 0, 0, &software);
				RegOpenKeyEx(software, "edp.org", 0, 0, &edp);
				RegOpenKeyEx(edp, "TriSolve", 0, KEY_QUERY_VALUE, &trisolve);

				// Get last options.
				length = sizeof newOptions;
				status = RegQueryValueEx(trisolve, "Options", 0, &type,
					(LPBYTE) &newOptions, &length);
				// If we successfully retrieved options, use them.
				if (status == ERROR_SUCCESS && type == REG_BINARY &&
						length == sizeof newOptions)
					info->options = newOptions;
				/*	When the options get more complicated, it may be a good
					idea to validate them, in something funny has been done
					to the registry.  Also, complicated options should have
					a dialog button to reset them to the original state.
				*/

				/*	Get name of last pieces file.  This may be "" to indicate
					the user closed the file, and we wish to restore that state.
				*/
				length = sizeof info->preview[piecesID].name;
				status = RegQueryValueEx(trisolve, "Pieces", 0, &type,
					(LPBYTE) info->preview[piecesID].name, &length);

				// If we did not retrieve a name, try the default.
				if (status != ERROR_SUCCESS || type != REG_SZ)
					_fullpath(info->preview[piecesID].name,
						"pieces.tri", sizeof info->preview[piecesID].name);

				/*	Get name of last goal file.  This may be "" to indicate
					the user closed the file, and we wish to restore that state.
				*/
				length = sizeof info->preview[goalID].name;
				status = RegQueryValueEx(trisolve, "Goal", 0, &type,
					(LPBYTE) info->preview[goalID].name, &length);

				// If we did not retrieve a name, try the default.
				if (status != ERROR_SUCCESS || type != REG_SZ)
					_fullpath(info->preview[goalID].name,
						"goal.tri", sizeof info->preview[goalID].name);

				RegCloseKey(trisolve);
				RegCloseKey(edp);
				RegCloseKey(software);
			}

			// If we have a pieces file name and it is not "", try to open it.
			if (info->preview[piecesID].name[0] != '\0')
				readFile(info, FALSE, piecesID);

			info->preview[piecesID].readonly = TRUE;
			info->preview[piecesID].modified = FALSE;
			enableCommands(window, info, piecesID);

			// If we have a goal file name and it is not "", try to open it.
			if (info->preview[goalID].name[0] != '\0')
				readFile(info, FALSE, goalID);
			
			info->preview[goalID].readonly = TRUE;
			info->preview[goalID].modified = FALSE;
			enableCommands(window, info, goalID);

			// Start in solve mode.
			setEditMode(window, info, FALSE);
			setMessage(info);

			// Pass options to the child display windows.
			SendMessage(info->preview[piecesID].window, WM_SETOPTIONS,
				0, (LPARAM) &info->options);
			SendMessage(info->preview[goalID].window, WM_SETOPTIONS,
				0, (LPARAM) &info->options);
			SendMessage(info->exhibit.window, WM_SETOPTIONS,
				0, (LPARAM) &info->options);

			return 0;
		}

		case WM_SYSCOLORCHANGE:
			SendMessage(info->status.window, WM_SYSCOLORCHANGE, 0, 0);
			return 0;

		case WM_SIZE:
		{
			RECT	rect, pieces, goal;
			LONG	width, height;


			GetClientRect(window, &rect);
			width = rect.right - rect.left;
			height = rect.bottom - rect.top;

			placePreviews(&pieces, &goal, width, height, widthStatus, heightStatus);

			MoveWindow(info->preview[piecesID].window,
				pieces.left, pieces.top, pieces.right, pieces.bottom, TRUE);
			MoveWindow(info->preview[goalID].window,
				goal.left, goal.top, goal.right, goal.bottom, TRUE);
			MoveWindow(info->exhibit.window, widthStatus, 0,
				width-widthStatus, height, TRUE);
	
			return 0;
		}

		case WM_COMMAND:
		{
			int		wmID, wmEvent;
			ChildID	ID;


			wmID    = LOWORD(wParam);
			wmEvent = HIWORD(wParam);

			// Make a note of the child for the Pieces and Goal commands.
			switch (wmID)
			{
				default:
				case IDA_PIECES:
				case IDM_PIECES_NEW:
				case IDM_PIECES_OPEN:
				case IDM_PIECES_CLOSE:
				case IDM_PIECES_SAVE:
				case IDM_PIECES_SAVEAS:
				case IDM_PIECES_REARRANGE:
					ID = piecesID;
					break;

				case IDA_GOAL:
				case IDM_GOAL_NEW:
				case IDM_GOAL_OPEN:
				case IDM_GOAL_CLOSE:
				case IDM_GOAL_SAVE:
				case IDM_GOAL_SAVEAS:
					ID = goalID;
					break;
			}

			// Parse the menu selections.
			switch (wmID)
			{
				// Accelerator to select status window.
				case IDA_STATUS:
				{
					Shape	*shape;


					SetFocus(info->status.window);
					SendMessage(info->status.window, WM_GETSHAPE, 0, (LPARAM) &shape);
					ChangeExhibit(info, MAKELONG(statusID, CE_IMPERATIVE), shape);
					return 0;
				}

				// Accelerators to select pieces and goal preview windows.
				case IDA_PIECES:
				case IDA_GOAL:
					ChangeExhibit(info, MAKELONG(ID, CE_IMPERATIVE),
						info->preview[ID].shape);
					// Fall through to set focus to exhibit window.

				// Accelerator to select exhibit window.
				case IDA_EXHIBIT:
					SetFocus(info->exhibit.window);
					return 0;

				/*	F6 and Control+F6 are defined as accelerator keys
					to send IDA_PANEFORWARD, which will cycle the input
					focus through panes.  Shift+F6 and Control+Shift+F6
					will cycle backward.  With only two panes (status
					and exhibit), these are equivalent.
				*/
				case IDA_PANEFORWARD:
				case IDA_PANEBACKWARD:
					SetFocus(GetFocus() == info->exhibit.window ?
						info->status.window : info->exhibit.window);
					return 0;

				case IDM_START_SOLVING:
				{
					if (info->status.problem == NULL)
					{
						// When there is no existent thread, try to start one.

						// Make sure there are pieces.
						if (info->preview[piecesID].shape == NULL)
						{
							DisplayError("TriSolve",
"Cannot start Solver because there are no pieces to use.\n\
Use Pieces New or Pieces Open to create or read pieces.");
							return 0;
						}

						// Make sure there's a goal.
						if (info->preview[goalID].shape == NULL)
						{
							DisplayError("TriSolve",
"Cannot start Solver because there is no goal to make.\n\
Use Goal New or Goal Open to create or read a goal shape.");
							return 0;
						}

						// Leave edit mode.
						setEditMode(window, info, FALSE);

						// Tell status window to free old information.
						SendMessage(info->status.window, WM_SOLVER_STARTING, 0, 0);

						// Set up new data for Solver.
						info->status.pieces =
							extractPieces(info->preview[piecesID].shape, TRUE);
						info->status.info.work.shape =
							copyShape(info->preview[goalID].shape);
						info->status.info.work.next = NULL;

						// Change display to the Solver work area.
						ChangeExhibit(info,
							MAKELONG(statusID, CE_REDRAW | CE_IMPERATIVE),
							info->status.info.work.shape);

						// Start the Solver.
						info->status.problem = StartSolver(window, info->status.pieces,
							&info->status.info.work);

						/*	If unsuccessful, emulate a WM_SOLVER_ENDING message.
							This is important, because we allocated resources which
							should be freed.
						*/
						if (info->status.problem == NULL)
							SendMessage(window, WM_SOLVER_ENDING, SOLVER_ERROR, 0);
						// If successful, change menu item to "Stop Solver".
						else
						{
							MENUITEMINFO		menuInfo =
								{ sizeof menuInfo, MIIM_TYPE, MFT_STRING };


							menuInfo.dwTypeData = "&Stop Solver";
							SetMenuItemInfo(GetSubMenu(GetMenu(window), POS_FILE_MENU),
								IDM_START_SOLVING, FALSE, &menuInfo);
						}
					}
					else
					{
						/*	When there is a Solver thread, tell it to stop, but
							check with user first.
						*/
						MessageBeep(MB_ICONQUESTION);
						if (IDOK == MessageBox(window,
								"Do you want to abort the Solver?",
								"TriSolve", MB_OKCANCEL))
							StopSolver(info->status.problem);
						// There's no menu update here.  We'll do that when
						// the thread tells us it is ending.
					}

					return 0;
				}
				
				case IDM_FILE_PRINT:
				{
					Shape	*shape;


					SendMessage(info->exhibit.window, WM_GETSHAPE, 0, (LPARAM) &shape);
					print(window, &info->options, shape);
					return 0;
				}

				case IDM_EXIT:
					PostMessage(window, WM_CLOSE, 0, 0);
					return 0;

				case IDM_PIECES_NEW:
				case IDM_GOAL_NEW:
					if (newFile(window, info, ID))
						setEditMode(window, info, TRUE);
					return 0;

				case IDM_PIECES_OPEN:
				case IDM_GOAL_OPEN:
					if (openFile(window, info, ID))
						setEditMode(window, info, FALSE);
					return 0;

				case IDM_PIECES_CLOSE:
				case IDM_GOAL_CLOSE:
					if (closeFileTest(window, info, ID))
					{
						closeFile(window, info, ID);
						/*	Set file name to null when user closes window by command,
							but not when it is closed while exiting application.
						*/
						TriSolveRegSet(info->preview[ID].caption, REG_SZ, (BYTE *) "", 1);
					}
					return 0;

				case IDM_PIECES_SAVE:
				case IDM_GOAL_SAVE:
					saveFile(window, info, ID);
					return 0;

				case IDM_PIECES_SAVEAS:
				case IDM_GOAL_SAVEAS:
					saveAs(window, info, ID);
					return 0;

				case IDM_PIECES_REARRANGE:
				{
					Pieces	*pieces;
					Shape	*shape;
					RECT	rect;


					// Extract the pieces from the shape.
					pieces = extractPieces(info->preview[ID].shape, TRUE);

					/*	Get an aggregate image of pieces tailored for current
						pieces window dimensions and assigned identities.
					*/
					GetClientRect(info->preview[ID].window, &rect);
					shape = aggregatePieces(pieces, rect.right, rect.bottom);
					freePieces(pieces);

					// If we succeeded, replace pieces.  Otherwise, make no change.
					if (shape != NULL)
					{
						freeShape(info->preview[ID].shape);
						info->preview[ID].shape = shape;
						ChangeDisplay(info, info->preview[ID].window,
							&info->preview[ID].info, info->preview[ID].shape);
					}

					return 0;
				}

				case IDM_VIEW_EDITMODE:
					setMessage(info);
					// If showing status display when entering edit mode, change to goal.
					if (info->exhibitView == statusID)
						ChangeExhibit(info, MAKELONG(goalID, CE_IMPERATIVE),
							info->preview[goalID].shape);
					SetFocus(info->exhibit.window);
					// Fall through to set edit mode.

				case IDM_VIEW_SOLVEMODE:
					setEditMode(window, info, wmID == IDM_VIEW_EDITMODE);
					return 0;

				case IDM_VIEW_FITINWINDOW:
					PostMessage(info->exhibit.window, WM_SETVIEW, VIEW_FIT, 0);
					return 0;

				case IDM_VIEW_STANDARDSIZE:
					PostMessage(info->exhibit.window, WM_SETVIEW, VIEW_FIXED, STANDARD_SIZE);
					return 0;

				case IDM_VIEW_ZOOMIN:
					PostMessage(info->exhibit.window, WM_SETVIEW, VIEW_ZOOMIN, 0);
					return 0;

				case IDM_VIEW_ZOOMOUT:
					PostMessage(info->exhibit.window, WM_SETVIEW, VIEW_ZOOMOUT, 0);
					return 0;

				case IDM_VIEW_ZOOMTO:
				{
					int	percentage;


					percentage = DialogBox(hInstance, (LPCSTR) IDD_ZOOMTO, window, ZoomTo);
					if (percentage == -1)
						DisplayWindowsError("TriSolve: Creating Zoom To dialog");
					else if (0 < percentage)
					{
						percentage = STANDARD_SIZE * percentage / 100;
						PostMessage(info->exhibit.window, WM_SETVIEW, VIEW_FIXED,
							percentage == 0 ? 1 : percentage);
					}
					return 0;
				}

				case IDM_TOOLS_CLEARHISTORY:
				{
					HKEY	software, edp;
					DWORD	nSubkeys, nValues;


					MessageBeep(MB_ICONQUESTION);
					if (IDOK != MessageBox(window,
"TriSolve remembers your options and last files in the Windows Registry. \
This will delete those entries, until you set options or select files again. \
TriSolve makes no other persistent changes in your system. Continue?",
						"TriSolve", MB_OKCANCEL | MB_ICONQUESTION))
						return 0;

					RegOpenKeyEx(HKEY_CURRENT_USER, "Software", 0, 0, &software);
					RegOpenKeyEx(software, "edp.org", 0, KEY_QUERY_VALUE, &edp);
					RegDeleteKeyRecursive(edp, "TriSolve");

					// Check edp.org key for subkeys and values.
					RegQueryInfoKey(edp, NULL, NULL, NULL, &nSubkeys, NULL,
						NULL, &nValues, NULL, NULL, NULL, NULL);
					RegCloseKey(edp);

					// Delete edp.org key if it is no longer in use.
					if (nSubkeys == 0 && nValues == 0)
						RegDeleteKey(software, "edp.org");

					RegCloseKey(software);
					return 0;
				}

				case IDM_TOOLS_OPTIONS:
				{
					OptionsDialogInfo	dialogInfo;
					int					result;


					// Pass current options to dialog.
					dialogInfo.options = info->options;

					result = DialogBoxParam(hInstance, (LPCSTR) IDD_OPTIONS,
						window, Options, (LPARAM) &dialogInfo);
					if (result == -1)
						DisplayWindowsError("TriSolve: Creating Options dialog");
					else if (result == IDOK)
					{
						// Update our options and inform display windows.
						info->options = dialogInfo.options;
						SendMessage(info->preview[piecesID].window, WM_SETOPTIONS,
							0, (LPARAM) &info->options);
						SendMessage(info->preview[goalID].window, WM_SETOPTIONS,
							0, (LPARAM) &info->options);
						SendMessage(info->exhibit.window, WM_SETOPTIONS,
							0, (LPARAM) &info->options);

						// Save options in registry.
						TriSolveRegSet("Options", REG_BINARY, (BYTE *) &info->options,
							sizeof info->options);
					}
					return 0;
				}

				case IDM_HELP_CONTENTS:
					if (!WinHelp(window, "TriSolve.hlp", HELP_FINDER, 0))
						DisplayWindowsError(helpTitle);
					return 0;

				case IDM_HELP_INDEX:
					if (!WinHelp(window, "TriSolve.hlp", HELP_PARTIALKEY, (DWORD) ""))
						DisplayWindowsError(helpTitle);
					return 0;

				case IDM_HELP_REGISTER:
					if (!WinHelp(window, "TriSolve.hlp>edp", HELP_CONTEXT, IDH_REGISTER))
						DisplayWindowsError(helpTitle);
					return 0;

				case IDM_ABOUT:
					if (-1 == DialogBox(hInstance, (LPCTSTR) IDD_ABOUT, window, About))
						DisplayWindowsError("TriSolve: Creating About dialog");
					return 0;
			}
			break;
		}

		case WM_SETSHAPE:
			/*	When preview or exhibit window is changed, set focus to exhibit
				window.  This is a bit of a kludge for two reasons.  First, the
				behavior we really want is that when a window is clicked, the
				input focus is set.  It happens that clicking a window sends
				WM_SETSHAPE, so we piggyback on the message.  Second, clicking
				on the preview windows sets focus to the exhibit window, so we
				have to customize that behavior here.  Giving the preview windows
				lives of their own would let them set their own focus and
				eliminate both of these kludges.
			*/
			if (LOWORD(wParam) != statusID)
				SetFocus(info->exhibit.window);

			/*	When the exhibit window notifies us of a change to the pieces or
				the goal, notify the corresponding preview window.
			*/
			if (LOWORD(wParam) == exhibitID &&
					(info->exhibitView == piecesID || info->exhibitView == goalID))
			{
				// If edit was made, note that shape has been modified.
				if (HIWORD(wParam) & CE_EDIT)
				{
					info->preview[info->exhibitView].modified = TRUE;
					setMessage(info);
					// Don't pass CE_EDIT to child when changing exhibit, below.
					wParam &= ~ MAKELONG(0, CE_EDIT);
				}

				// Exhibit window changed.  Tell corresponding preview window.
				PostMessage(info->preview[info->exhibitView].window,
					WM_SETSHAPE, wParam, lParam);
			}
			/*	When another child informs us of a change, notify the
				exhibit window.
			*/
			else
				ChangeExhibit(info, wParam, (Shape *) lParam);
			return 0;

		case WM_SOLUTION_FOUND:
			// Forward to the status window.
			SendMessage(info->status.window, WM_SOLUTION_FOUND, 0, 0);
			return 0;

		case WM_SOLVER_ENDING:
		{
			MENUITEMINFO		menuInfo = { sizeof menuInfo, MIIM_TYPE, MFT_STRING };


			// Release the pieces the Solver was using.
			freePieces(info->status.pieces);
			info->status.pieces = NULL;

			// Release the memory the Solver was using for its own context.
			free(info->status.problem);
			info->status.problem = NULL;

			/*	Note that the work area and solutions are not freed now.
				They remain available for viewing until they are freed
				when a new problem is started.
			*/

			// Change menu item to start Solver.
			menuInfo.dwTypeData = "&Start Solver";
			SetMenuItemInfo(GetSubMenu(GetMenu(window), POS_FILE_MENU),
				IDM_START_SOLVING, FALSE, &menuInfo);
			info->status.problem = NULL;

			// Tell the status window.
			SendMessage(info->status.window, WM_SOLVER_ENDING, wParam, 0);
			return 0;
		}

		case WM_CLOSE:
			if (info->status.problem != NULL)
			{
				MessageBeep(MB_ICONQUESTION);
				if (IDOK != MessageBox(window,
						"The Solver is running. Do you want to exit anyway?",
						"TriSolve", MB_OKCANCEL))
					return 0;
			}
			if (	!closeFileTest(window, info, piecesID) ||
					!closeFileTest(window, info, goalID))
				return 0;
			closeFile(window, info, piecesID);
			closeFile(window, info, goalID);
			DestroyWindow(window);
			return 0;

		case WM_DESTROY:
			WinHelp(window, NULL, HELP_QUIT, 0);
			PostQuitMessage(0);
			return 0;
	}

	return DefWindowProc(window, message, wParam, lParam);
}


// Register the TriSolve window class.
static ATOM	TriSolveRegister(HINSTANCE hInstance)
{
	static WNDCLASSEX	wcex = {
		sizeof wcex, CS_HREDRAW | CS_VREDRAW, TriSolveWindow, 0, 0,
		0, 0, 0, (HBRUSH) (COLOR_WINDOW+1), (LPCSTR) IDC_TRISOLVE,
		"TriSolve", NULL
	};

	HICON		icon;
	ATOM		result;
	

	icon = LoadIcon(hInstance, (LPCTSTR) IDI_TRISOLVE);
	if (icon == NULL)
		DisplayWindowsError("TriSolve: Loading icon");

	wcex.hInstance		= hInstance;
	wcex.hIcon			= icon;
	wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);

	result = RegisterClassEx(&wcex);
	if (result == 0)
		DisplayWindowsError("TriSolve: Registering TriSolve window class");

	return result;
}


// The main routine for the application.
int APIENTRY	WinMain(HINSTANCE hInstance,
	HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	TriSolveInfo	info;

	MSG		message;
	HACCEL	accelTable;
	HWND	window;
	int		t;


	// Create window classes.
	if (0 == TriSolveRegister(hInstance))
		return 0;
	if (0 == TriDisplayRegister(hInstance))
		return 0;

	/*	TriSolve will initialize its own data, but we use one
		member to dispatch messages to a dialog window, so we
		need that initialized right away.
	*/
	info.status.window = NULL;

	window = CreateWindow("TriSolve", "TriSolve",
		WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN,
		CW_USEDEFAULT, 0, CW_USEDEFAULT, 0, NULL, NULL, hInstance,
		(void *) &info);
	if (window == NULL)
	{
		DisplayWindowsError("TriSolve: Creating main window");
		return 0;
	}

	ShowWindow(window, nCmdShow);

	accelTable = LoadAccelerators(hInstance, (LPCTSTR) IDC_TRISOLVE);
	if (accelTable == NULL)
	{
		DisplayWindowsError("TriSolve: Loading accelerators");
		return 0;
	}

	while (1 == (t = GetMessage(&message, NULL, 0, 0)))
	{
		// Translate accelerator keys globally for application.
		if (TranslateAccelerator(window, accelTable, &message))
			;
		// Dispatch messages for TriSolve's dialog window (yuck).
		else if (info.status.window != NULL &&
				IsDialogMessage(info.status.window, &message))
			;
		// Dispatch regular messages.
		else 
		{
			TranslateMessage(&message);	// Translate keys into characters.
			DispatchMessage(&message);
		}
	}
	if (t == -1)
		DisplayWindowsError("TriSolve: GetMessage");

	return message.wParam;
}