/*	Copyright 2002 by Eric Postpischil, http://edp.org.
	See license information in index.html.
*/


#include	"StdAfx.h"
#include	"resource.h"

#include	<math.h>	// for floor


/*	A note about our use of nPos and nTrackPos in the SCROLLINFO
	structures:  We don't use nTrackPos to communicate with
	Windows, so we adopt it for another purpose.  nPos will
	record where the thumb appears and represents the location
	the user desires to see.  nTrackPos records where we are
	currently drawing.  When nPos is changed, we set a timer.
	When WM_TIMER messages are received, we move nTrackPos
	closer to nPos, in a smoothly accelerated way, giving a
	pleasing smooth scrolling.
*/
  
/*	We have four coordinate systems to deal with:

		index space
			Index space uses integer coordinates to index
			elements in rectangular arrays.  Each element
			represents a triangle.  Visualize a row of
			equilateral triangles alternately pointing up
			and down, then visualize many such rows.  The
			first integer dimension selects a row, and the
			second integer dimension selects a triangle
			within the row.

		Cartesian space
			The triangles are given a side of unit length,
			resulting in a height of sqrt(3)/2.  So the
			distance from row to row is sqrt(3)/2, defined
			in HEIGHT.  Traveling within a row, two
			triangles must be traversed to move the distance
			of one side, because the triangles are nestled
			together alternately pointing up and down.
			Thus the horizontal distance from the center of
			one triangle to the center of next is 1/2,
			defined in WIDTH.

		page space
			Page space is the coordinate system used in
			passing coordinates to Windows GDI routines,
			in the absence of world-space.  Since it uses
			integer coordinates, we select a large scaling
			from Cartesian space to page space to maximize
			precision.

		device space
			Device space is measured in pixels.  Scroll
			ranges and positions are maintained in device
			space.  The scroll position is passed to the
			GDI routines by setting the viewport origin.
*/

/*	The manual says page space is 2^32 units wide, but testing
	shows that signed 16-bit coordinates are used.
*/
#define	COORDINATE_BOUND	32767

/*	Define the margin (in device space) to be allowed around
	the image.
*/
#define	MARGIN	2


// Array of color values for coloring pieces.  Initialized at class registration.
static COLORREF	color[NUM_COLORS];


/*	getShapeRect provides the bounding box that encloses the
	graphic image of a shape.  The values are in index space
	coordinates.  Within a shape, triangles are identified
	by a single point representing a triangle origin, but the
	actual triangle contains points extending two units right
	and one unit down.  The bounding box includes these points.

	If margin is TRUE, additional space is provided around
	the actual image, to give the user additional space to
	draw while in edit mode.
*/
static inline RECT	getShapeRect(Shape *shape, Bool margin)
{
	RECT	rect;


	SetRect(&rect, 0, 0, shape->width+1, shape->height);

	if (margin)
		InflateRect(&rect, max(2, 4-shape->width/2), max(1, 4-shape->height/2));

	return rect;
}


/*	Set the scroll bar ranges.  This works for either dimension;
	the names leftmost and rightmost apply equally as topmost
	and bottommost.

	leftmost and rightmost give the scroll range in device space.
	Like many GDI ranges, these specify a half-open interval.
	However, the nMax member of SCROLLINFO is inclusive, not
	exclusive, so it is one less than rightmost.
*/
static inline void	setRange(ScrollInfo *scroll, int leftmost, int rightmost)
{
	int			t;


	scroll->scroll.nMin = leftmost;
	scroll->scroll.nMax = rightmost - 1;

	/*	The page is the lesser of all the pixels we have to display
		or what fits in the window.
	*/
	scroll->scroll.nPage = min(rightmost - leftmost, scroll->length);
	
	/*	The difference between the display length and the page
		length is unused space in the display.  Divide it in
		half and move the scroll range by that amount to center
		the page in the display.  (We shift it by the larger "half"
		just to decrease the rightmost coordinate a little, since
		the left side has more room to grow without hitting the
		coordinate bounds.)
	*/
	t = LARGER_HALF(scroll->length - scroll->scroll.nPage);
	scroll->scroll.nMin	-= t;
	scroll->scroll.nMax	-= t;

	/*	nMax is the maximum pixel to be displayed.  The maximum
		position the top of the page can be set to is that that
		puts the maximum pixel at the bottom of the page.
	*/
	scroll->posMax = scroll->scroll.nMax - scroll->scroll.nPage + 1;
}


static inline float	getProportion(int position, int minimum, int maximum)
{
	return minimum == maximum ? .5f :
		(position - minimum) / (float) (maximum - minimum);
}


static inline int	applyProportion(float proportion, int minimum, int maximum)
{
	return (int) ((proportion * (maximum - minimum)) + .5) + minimum;
}


/*	setSubScales sets the index-to-page and page-to-device scales,
	given the index-to-device scale.
	
	The leftmost and rightmost pixels of the image are used to
	set the page-to-device scale.  The index-to-page scale
	follows as a consequence.  (Like many GDI ranges, the
	leftmost to rightmost range is half-open, excluding rightmost.)

	Then the scroll information is reset for the new scaling
	and image bounds.  If window is NULL, no attempt is made
	to set its scroll information, although the data structures
	are filled in.

	This routine works for each dimension, replacing leftmost
	and rightmost with topmost and bottommost.

	The updateType parameter describes how the new scroll
	position is set.  If it is FALSE, the new scroll position has
	the same ratio of nPos in its allowed range as the old scroll
	position.  This is reasonably pleasant when changing the scaling,
	the window size, and so on.  If it is TRUE, the new scroll
	position has the same pixel offset from the image center as the
	old scroll position.  This is useful during editing when
	addition of triangles expands the shape but leaves the center
	unchanged.

	In the future, it might be good to generalize this by adding a
	concept of current position within the shape and then to maintain
	that position at the same point in the display through a change
	in scaling.  That would serve both with scale/size changes and
	edit changes.  Perhaps it would even give cleaner code.  The
	current position might be recorded as a floating-point location
	in index-space, which gives a shape-invariant reference yet is
	more precise than a triangle quantum.  Still, that might result
	in single-pixel fluctuations due to rounding, so the arithmetic
	will have to be explored during implementation.
*/
static void	setSubScales(HWND window, ScrollInfo *scroll,
	float indexToDevice, int leftmost, int rightmost, Bool updateType)
{
	/*	extentDevice records the device-space coordinate of the rightmost
		pixel we will ever display.  If the image extends beyond one
		window, this coordinate is rightmost, which will be displayed
		when the scroll is at its rightmost position.  If the image does
		not fill the window, there will be a margin between the rightmost
		pixel of the image and the rightmost pixel of the display.  This
		margin is half the difference between the display length and the
		distance spanned by the image.
	*/
	scroll->extentDevice = rightmost - 1 +
		max(0, SMALLER_HALF(scroll->length - (rightmost - leftmost)));

	/*	extentDevice is used to set the viewport extent.  We set the
		window extent to COORDINATE_BOUND, so we have now determined
		the page-to-device scaling in this dimension.  COORDINATE_BOUND
		scales to extentDevice, so the scale ratio is
		COORDINATE_BOUND / extentDevice.
	*/

	/*	The index-to-page scaling follows arithmetically from the
		index-to-device and page-to-device scalings.
	*/
	scroll->indexToPage = (int) (indexToDevice *
		COORDINATE_BOUND / scroll->extentDevice);

	/*	indexToPage can reach zero, in which case we set it to 1 to
		avoid divide-by-zero errors elsewhere.  It is commonly zero when
		a window is reduced to zero length, and so we don't care about
		the scaling.  It can also become zero when the scaling is so
		disproportionate that each triangle ought to be mapped to less
		than one page unit.  We're raising it to a page unit.  That can't
		cause overflow errors, since index units are limited to short
		values just like coordinates, so multiplying by 1 does not
		produce excessive values.  However, it means the value of
		rightmost that is used for the maximum scroll range will not
		reach the rightmost image position in the new scale.  That
		does not really matter, since the display at this scaling is
		so disproportionate as to be unusable.
	*/
	if (scroll->indexToPage <= 0)
		scroll->indexToPage = 1;

	// Set scroll ranges while preserving relative scroll position.
	{
		float	relativePosition;
		
		
		// See comments above about updateType.
		relativePosition = updateType
			? (float) (scroll->scroll.nTrackPos -
				(scroll->scroll.nMin + scroll->posMax)/2)
			: getProportion(scroll->scroll.nTrackPos,
				scroll->scroll.nMin, scroll->posMax);

		setRange(scroll, leftmost, rightmost);

		// Restore relative scroll position.
		scroll->scroll.nPos =
		scroll->scroll.nTrackPos = updateType
			? (int) (relativePosition + (scroll->scroll.nMin + scroll->posMax)/2)
			: applyProportion(relativePosition, scroll->scroll.nMin, scroll->posMax);
	}

	/*	Note that we do not reset the scroll velocity.  If the
		user changes shapes while we are moving, the new shape
		will appear in the display already moving at the old
		velocity.  We'll decelerate and return it to the desired
		position.

		Resetting velocity here would prevent that, but it is fun.
		There is a possibility the old velocity and the new
		position could exceed the coordinate bounds, but it really
		is fun, and the position will correct itself.
	*/

	if (window != NULL)
		SetScrollInfo(window, scroll->bar, &scroll->scroll, TRUE);
}


/*	Calculate the desired coordinate scaling.

	The index-to-Cartesian scaling is fixed.  The width of
	a triangle is two units in index space and WIDTH in
	Cartesian space.  The height is one unit in index space
	and HEIGHT in Cartesian space.

	So, first, the target Cartesian-to-device scaling is set.

	In fixed-view mode, the target is to scale one triangle
	width to the specified number of pixels, but not so large
	that coordinates in device space would exceed COORDINATE_BOUND.

	In fit-to-window mode, the target is the largest scale
	that maps the shape within the display in both width and
	height.

	Then we set the index-to-page and page-to-device
	scalings.  We will use an anisotropic map with
	separate values for these scalings, which are stored in
	structures named horz and vert.  Because both dimensions
	have the same Cartesian-to-device scaling, the final
	result is isotropic, up to rounding errors.

	For speed and convenience, the Cartesian-to-page
	and page-to-device scalings are represented by values
	stored in structure members indexToPage and
	extentDevice:

		Index-to-page scaling is indexToPage.
		This makes converting index-space values to
		page-space values easy -- just multiply by
		indexToPage.

		Page-to-device scaling is COORDINATE_BOUND /
		extentDevice.  During drawing, we only
		need to set the transformation by passing
		extentDevice to SetViewportExtEx.  At
		other times, we need the more complicated
		expression, but these are outside of the
		drawing routines that we would like to be
		fast.

	Note that the page-to-device scaling is not the
	integer expression COORDINATE_BOUND / extentDevice.
	For example, y * (COORDINATE_BOUND / extentDevice)
	does not generally give the correct value of y in
	device space.  The correct expression is
	(y * COORDINATE_BOUND) / extentDevice.  The
	truncation error could be significant and visible and
	could even cause overflow errors.

	After scaling is set, scrolling information for the given
	window is set.  window may be NULL, which allows this
	routine to be used to set scaling for printer device
	contexts, which do not have scroll bars.  Later, perhaps
	we could separate the scaling and scrolling into separate
	routines.

	See setSubScales for information on the updateType parameter.
*/
static void	setScaling(HWND window, TriDisplayInfo *info, Bool updateType)
{
	const int	marginBefore =
					- MARGIN - SMALLER_HALF(info->options.perimeterPenPixels),
				marginAfter =
					+ MARGIN + LARGER_HALF(info->options.perimeterPenPixels);
	/*	The pen thickness is divided as evenly as possible before
		and after the central location.  We do not actually have
		control over which side of the center the extra pixel will
		fall, but the MARGIN will absorb that.
	*/

	Shape	* const shape = info->shape;
	RECT	rect;
	float	CartesianToDevice;


	// Erase window (if any).
	if (window != NULL)
		InvalidateRect(window, NULL, TRUE);

	if (shape == NULL)
		return;

	// Find index-space bounds of shape as it will be drawn.
	rect = getShapeRect(shape, info->editMode);

	/*	If the user requests fit-to-window, we use it all the time in Solve
		Mode.  In Edit Mode, we use it only once to calculate the scaling,
		which then remains fixed during the edit session.
	*/
	if (info->editMode ? info->viewSize == 0 : info->viewMode == VIEW_FIT)
	{
		// Find width and height of shape in Cartesian space.
		const float	widthCartesian = (rect.right-rect.left) * WIDTH,
					heightCartesian = (rect.bottom-rect.top) * HEIGHT;

		// Find width and height of display available for shape proper.
		const int	totalMargin = 2*MARGIN + info->options.perimeterPenPixels;
		const int	widthDisplay = info->horz.length - totalMargin,
					heightDisplay = info->vert.length - totalMargin;


		/*	Set Cartesian-to-Device scaling to smaller of scaling that
			keeps width in display or scaling that keeps height in display.
		*/
		CartesianToDevice =
			widthDisplay * heightCartesian <= heightDisplay * widthCartesian
			? widthDisplay / widthCartesian
			: heightDisplay / heightCartesian;
		/*	Note that heightCartesian can be zero, when a shape has no points.
			widthCartesian is at least one (see getShapeRect), so, when
			heightCartesian is zero, we want to be sure the first ratio is
			evaluated, not the second.
		*/

		// For edit mode, calculate the fixed scale to use.
		if (info->editMode)
			info->viewSize = (int) (2 * WIDTH * CartesianToDevice);
	}

	// Use a fixed-scale view when requested or while in Edit Mode.
	if (info->viewMode == VIEW_FIXED || info->editMode)
	{
		const float	biggestCartesian =
			max(rect.right * WIDTH, rect.bottom * HEIGHT);


		/*	Set Cartesian-to-Device scaling to smaller of requested scaling
			or largest scaling that keeps device coordinates in bound.
		*/
		CartesianToDevice = min(info->viewSize / (2 * WIDTH),
			(COORDINATE_BOUND - marginAfter) / biggestCartesian);
	}

	/*	Having the Cartesian-to-device scaling, set the
		index-to-page and page-to-device.
	*/
	setSubScales(window, &info->horz, WIDTH * CartesianToDevice,
		(int) (marginBefore + rect.left * WIDTH * CartesianToDevice),
		(int) (marginAfter + rect.right * WIDTH * CartesianToDevice),
		updateType);
	setSubScales(window, &info->vert, HEIGHT * CartesianToDevice,
		(int) (marginBefore + rect.top * HEIGHT * CartesianToDevice),
		(int) (marginAfter + rect.bottom * HEIGHT * CartesianToDevice),
		updateType);

	/*	Record the pen thickness in page space.  Round up when converting so
		that truncation back to device space does not lose a pixel.
	*/
	info->perimeterPenThickness = (int) ceil(
		info->options.perimeterPenPixels * COORDINATE_BOUND /
		(float) info->horz.extentDevice);

	// Use triangle height for "line" scrolling, both vertically and horizontally.
	info->lineIncrement = info->vert.indexToPage * info->vert.extentDevice
		/ COORDINATE_BOUND;
	// Idea:  Use separate increments for vertical and horizontal?
	// Idea:  Set to minimum of triangle height or page increment?
}


/*	Process a change to the shape being displayed.
	See setSubScales for information on the updateType parameter.
*/
static void	setShape(HWND window, TriDisplayInfo *info, Bool redraw, Bool updateType)
{
	/*	If the shape bounds changed, recalculate scaling and
		redraw everything.  If not, invalidate the window to
		redraw the pieces within the shape but leave the background.
	*/
	if (redraw)
		setScaling(window, info, updateType);
	else
		InvalidateRect(window, NULL, FALSE);
}


/*	This is a utility routine for the WM_TIMER code.  It updates
	the velocity and the painting position and returns the amount
	to scroll the window in this update.  n gives the number of
	updates to apply.  (At least one update is apply on each call.)
*/
static int	smoothScroll(HWND window, TriDisplayInfo *info,
	ScrollInfo *scrollInfo, int n)
{
	const int	desired = scrollInfo->scroll.nPos,
				previous = scrollInfo->scroll.nTrackPos;

	int	velocity = scrollInfo->velocity,
		current = previous,
		t;


	do {
		// Figure out which way we want to go.
		if (current < desired)
		{
			/*	Where will we end if we accelerate now and then decelerate
				uniformly at 1 pixel per update per update?  (This is only
				valid if velocity is positive.  If velocity is negative,
				we need to accelerate in the positive direction.)
			*/
			t = current + (velocity+1) * (velocity+2) / 2;

			// If we can accelerate without overshooting, do so.
			if (velocity < 0 || t <= desired)
				velocity++;
			/*	What about using the current velocity now and declerating
				uniformly beginning in the next update?  If that overshoots,
				then decelerate now.
			*/
			else if (desired < t - (velocity+1))
				velocity--;
		}
		// Repeat the above modified for the other direction.
		else
		{
			t = current - (velocity-1) * (velocity-2) / 2;
			if (0 < velocity || desired <= t)
				velocity--;
			else if (t - (velocity-1) < desired)
				velocity++;
		}

		// Apply velocity.
		current += velocity;

	// Repeat for as many updates as we were told.
	} while (0 < --n);

	scrollInfo->velocity = velocity;
	scrollInfo->scroll.nTrackPos = current;

	// Tell caller how much to scroll window.
	return previous - current;
}


/*	That ends the coordinate transformation and scroll bar routines.
	Next we have the actual drawing routines.
*/


/*	Draw the triangle at the given coordinates.  scaleWidth and
	scaleHeight give the index-to-page scalings.
*/
static inline void	drawTriangle(HDC hdc, int x, int y, int parity,
	 int scaleWidth, int scaleHeight)
{
	POINT	triangle[3];
	int		t;


	//	The x coordinates of the vertices are evenly spaced.
	t  = x*scaleWidth;	triangle[0].x = t;
	t +=   scaleWidth;	triangle[1].x = t;
	t +=   scaleWidth;	triangle[2].x = t;

	/*	The y-coordinates of the vertices are down-up-down
		or up-down-up depending on which way the triangle points.
	*/
	t = y*scaleHeight;
	if (x+y+parity & 1)
	{
		triangle[1].y = t;

		triangle[0].y =
		triangle[2].y = t + scaleHeight;
	}
	else
	{
		triangle[0].y =
		triangle[2].y = t;

		triangle[1].y = t + scaleHeight;
	}

	Polygon(hdc, triangle, 3);
}


/*	Draw the left-side diagonal line of the triangle at
	the given coordinates.  scaleWidth and scaleHeight
	give the index-to-page scalings.
*/
static inline void	drawDiagonal(HDC hdc, int x, int y, int parity,
	 int scaleWidth, int scaleHeight)
{
	POINT	segment[2];
	int		t;


	parity ^= x+y & 1;	// Select orientation of the diagonal line (/ or \).

	t = x*scaleWidth;
	segment[0].x = t;
	segment[1].x = t + scaleWidth;

	t = y*scaleHeight;
	segment[  parity].y = t;
	segment[1-parity].y = t + scaleHeight;

	Polyline(hdc, segment, 2);
}


/*	Draw the horizontal line of the downward-pointing triangle
	at the given coordinates.  scaleWidth and scaleHeight give
	the index-to-page scalings.
*/
static inline void	drawHorizontal(HDC hdc, int x, int y,
	int scaleWidth, int scaleHeight)
{
	POINT	segment[2];
	int		t;


	t = x * scaleWidth;

	segment[0].x = t;
	segment[1].x = t + 2 * scaleWidth;

	segment[0].y =
	segment[1].y = y * scaleHeight;

	Polyline(hdc, segment, 2);
}


/*	Toggle the presence of a triangle in a shape, given its display
	coordinates.  If click is true, this is a command we must obey.
	Otherwise, it is the result of dragging, and we do not want to
	invert the triangle unless we have moved from the previous edit
	location.
*/
static void	invertTriangle(HWND window, TriDisplayInfo *info,
	unsigned short x, unsigned short y, Bool click)
{
	const int	scaleWidth = info->horz.indexToPage,
				scaleHeight = info->vert.indexToPage;

	HDC		hdc;
	POINT	point;
	Shape	*shape;


	/*	If we are not commanded to invert and cursor has not moved
		since last inversion, do nothing.  This eliminates the double
		inversion caused by a WM_MOUSEMOVE immediately after a
		WM_LBUTTONDOWN.  (We could just do the inversion on WM_MOUSEMOVE
		and not WM_LBUTTONDOWN, but the documentation does not
		guarantee we will receive a WM_MOUSEMOVE after each WM_LBUTTONDOWN,
		although we appear to.)

		Another check with index-space coordinates further down prevents
		double inversions from caused by movements of the cursor within
		the current triangle.  That check does not suffice for this case
		because a triangle inversion can change the shape bounds and
		cause a redraw that changes the index-to-device map.

		Moving the mouse can also do that, so it would be nice to find
		some behavior that gives the user easy access to triangles beyond
		the current shape bounds without undermining the index-to-device
		map while edits are being made.
	*/
	if (!click && x == info->lastEditDevice.x && y == info->lastEditDevice.y)
		return;
	info->lastEditDevice.x = x;
	info->lastEditDevice.y = y;
	
	shape = info->shape;

	if (shape == NULL)
		return;

	/*	If location is outside our window, do nothing.  This can
		happen when the user drags the cursor outside the window.
		Since we capture the mouse, we get WM_MOUSEMOVE messages
		as the cursor moves outside the window.
	*/
	if (x < 0 || info->horz.length <= x || y < 0 || info->vert.length <= y)
		return;

	hdc = GetDC(window);

	// Set prepared coordinate transformation.
	SetMapMode(hdc, MM_ANISOTROPIC);
	SetWindowExtEx(hdc, COORDINATE_BOUND, COORDINATE_BOUND, NULL);
	SetViewportOrgEx(hdc,
		-info->horz.scroll.nTrackPos, -info->vert.scroll.nTrackPos, NULL);
	SetViewportExtEx(hdc, info->horz.extentDevice, info->vert.extentDevice, NULL);

	// Convert device-space coordinates to page space.
	point.x = x;
	point.y = y;
	DPtoLP(hdc, &point, 1);

	// Convert page-space coordinates to index space.
	{
		float	xp, yp;


		/*	Figuring out which triangle covers a point is fun.  The
			y coordinate is easy, since the triangles are in horizontal
			rows.  Just take the page space coordinate, divide by the
			row height, and round down.  (The row height is
			the height in page space of one row in index space, which
			we have in scaleHeight.)

			The x coordinate is harder, since the triangles have
			diagonal lines, but our coordinates are rectangular.
			The left and right edges of the triangles form two sets
			of parallel lines, one set satisfying:

				y = + scaleHeight/scaleWidth * x - scaleHeight * 2*i,

			and the other set satisfying:

				y = - scaleHeight/scaleWidth * x + scaleHeight * 2*i.

			The term containing i is 2*i because there is one of
			each direction of diagonal line every two triangles,
			since the triangles point alternately upward and downward.

			Solving these equations for i gives an expression that
			maps a point (x, y) on any of the lines to the value of
			i for that line in the set.  Taking the floor of those
			expressions gives the i for the first line to the left of
			an arbitrary point (x, y) that may not be on any of the
			lines.  The floors of the two expressions are:

				floor(x / (2*scaleWidth) - y / (2*scaleHeight)) and
				floor(x / (2*scaleWidth) + y / (2*scaleHeight)).

			As a point (x, y) is moved right (or left), each of the
			floor expressions increases by one each time the point
			crosses the corresponding diagonal line.  The sum of the
			two expressions increases by one each time the point
			crosses any diagonal line.

			Furthermore, at x = 0, the sum of the two floor expressions
			is zero, by symmetry.  Thus, going right from x = 0 to any
			point, the sum of the two expressions is the number of
			triangles crossed, and hence is the index-space x coordinate.

			The above was all written when only even-parity reference
			frames were supported.  For an odd-parity frame, the y values
			shift by scaleHeight, with the result of changing the
			floor argument by 1/2, as done below.
		*/
		xp = (float) point.x / (2*scaleWidth);
		yp = (float) point.y / (2*scaleHeight);
		if (shape->parity)
			yp += .5;
		point.x = (int) (floor(xp-yp) + floor(xp+yp));

		point.y = (int) floor((float) point.y / scaleHeight);
	}

	// Invert triangle if we are commanded or it is new.
	if (click || info->lastEditIndex.x != point.x || info->lastEditIndex.y != point.y)
	{
		Point	change;


		// Remember last triangle edited.
		info->lastEditIndex.x = (short) point.x;
		info->lastEditIndex.y = (short) point.y;

		// Invert triangle.
		change = toggleTriangle(shape, point.x, point.y);
		if (change.x != 0 || change.y != 0)
		{
			// The origin moved, so translate coordinates to new frame.
			info->lastEditIndex.x += change.x;
			info->lastEditIndex.y += change.y;

			/*	Since the shape's boundaries have changed, we have to
				redo all the scale and scroll information.
			*/
			setShape(window, info, TRUE, TRUE);
		}
		else
		{
			/*	We do not need to redraw the whole shape.  Just
				invalidate a rectangle around the changed triangle,
				including the perimeter line.
			*/
			RECT		rect;
			unsigned	thickness;


			// Find bounding rectangle of triangle in page space.
			rect.left	= (point.x  ) * scaleWidth;
			rect.right	= (point.x+2) * scaleWidth;
			rect.top	= (point.y  ) * scaleHeight;
			rect.bottom	= (point.y+1) * scaleHeight;

			// Convert to device space.
			LPtoDP(hdc, (POINT *) &rect, 2);

			// Expand to account for perimeter line.
			thickness = LARGER_HALF(info->options.perimeterPenPixels);
			InflateRect(&rect, thickness, thickness);

			// Invalidate that portion of display.
			InvalidateRect(window, &rect, TRUE);
		}

		// Inform parent we have made an edit.
		PostMessage(GetParent(window), WM_SETSHAPE,
			MAKELONG(GetWindowLong(window, GWL_ID), CE_EDIT | CE_REDRAW),
			(LPARAM) info->shape);
	}

	ReleaseDC(window, hdc);
}


/*	Draw an edit guide line with slope rise/run passing through (x, y).
	Clip the endpoints for the 16-bit GDI.
*/
static inline void	drawGuide(HDC hdc, int run, int rise, int x, int y)
{
	POINT	segment[2];
	int		xT, yT, xB, yB;


	/*	Figure the coordinates of the top endpoint
		as clipped at y = -COORDINATE_BOUND.
	*/
	yT = -COORDINATE_BOUND;
	xT = (yT-y) * run / rise + x;

	// If necessary, clip top endpoint at left or right bounds.
	if (xT < -COORDINATE_BOUND)
	{
		xT = -COORDINATE_BOUND;
		yT = (xT-x) * rise / run + y;
	}
	else if (+COORDINATE_BOUND < xT)
	{
		xT = +COORDINATE_BOUND;
		yT = (xT-x) * rise / run + y;
	}

	/*	Figure the coordinates of the bottom endpoint
		as clipped at y = +COORDINATE_BOUND.
	*/
	yB = +COORDINATE_BOUND;
	xB = (yB-y) * run / rise + x;

	// If necessary, clip bottom endpiont at left or right bounds.
	if (xB < -COORDINATE_BOUND)
	{
		xB = -COORDINATE_BOUND;
		yB = (xB-x) * rise / run + y;
	}
	else if (+COORDINATE_BOUND < xB)
	{
		xB = +COORDINATE_BOUND;
		yB = (xB-x) * rise / run + y;
	}

	segment[0].x = xT;
	segment[0].y = yT;
	segment[1].x = xB;
	segment[1].y = yB;
	Polyline(hdc, segment, 2);
}


/*	Draw the shape.
	clip gives device-space coordinates of clip rectangle.
*/
static void	drawShape(HDC hdc, TriDisplayInfo *info, RECT clip)
{
	const int	scaleWidth = info->horz.indexToPage,
				scaleHeight = info->vert.indexToPage;
	const Shape	* const shape = info->shape;

	HPEN		pen;
	int			x, y, parity;


	if (shape == NULL)
		return;

	parity = shape->parity;

	// Set prepared coordinate transformation.
	SetMapMode(hdc, MM_ANISOTROPIC);
	SetWindowExtEx(hdc, COORDINATE_BOUND, COORDINATE_BOUND, NULL);
	SetViewportOrgEx(hdc,
		-info->horz.scroll.nTrackPos, -info->vert.scroll.nTrackPos, NULL);
	SetViewportExtEx(hdc, info->horz.extentDevice, info->vert.extentDevice, NULL);

	/*	Triangles whose perimeter lines intersect the update region
		must be redrawn, so expand the region of interest by distance
		from center of perimeter line to its edge.
	*/
	{
		 const unsigned	thickness = LARGER_HALF(info->options.perimeterPenPixels);


		 InflateRect(&clip, thickness, thickness);
	}

	/*	Below, we convert the device-space bounds of
		the clip region to index-space bounds of the
		triangles that must be redrawn.  We do this in
		two steps, devic space to page space and then
		page space to index space.
	*/

	// Convert device-space bounds to page-space.
	DPtoLP(hdc, (LPPOINT) &clip, 2);

	/*	Convert page-space bounds to index space.  The left and
		top bounds are rounded down.  The right and bottom
		bounds are rounded up, to be sure we enclose the entire
		area that must be painted.
	*/
	clip.left	/= scaleWidth;
	clip.right	 = (clip.right-1) / scaleWidth + 1;
	clip.top	/= scaleHeight;
	clip.bottom	 = (clip.bottom-1) / scaleHeight + 1;

	/*	Now clip contains bounds of the points that must be
		repainted.  However, the triangle with coordinates (x, y)
		has points that extend to (x+2, y+1).  (This is the
		bottom-right corner of an upward-pointing triangle.  If
		the triangle is downward-pointing, it has no point there,
		but it does have points as far right as x+2 and other
		points as far down as y+1.)

		We want clip to give us the indices of the triangles
		that must be repainted, so we'll adjust it.
	*/
	clip.left -= 2;
	clip.top -= 1;

	// In edit mode, draw guide lines.
	if (info->editMode)
	{
		POINT	segment[2];
		RECT	rect;
		int		i;


		// Guide lines are light gray.
		pen = (HPEN) SelectObject(hdc,
			CreatePen(PS_SOLID, 0, RGB(192, 192, 192)));

		// Align borders to triangle vertices.
		rect = clip;
		/*	If right-top corner is not at a vertex, raise top.  This avoids
			making the right bound larger.  It might be near the
			coordinate bound.  The top is at worst a little less than zero
			and has plenty of room to move.
		*/
		if (rect.right+rect.top+parity & 1)
			rect.top--;
		/*	If the left-top corner is not at a vertex, move the left border.
			It also has room to move, and we can't move the top without
			upsetting the right-top corner.
		*/
		if (rect.left+rect.top+parity & 1)
			rect.left--;

		// Draw horizontal guide lines.
		segment[0].x = -COORDINATE_BOUND;
		segment[1].x = +COORDINATE_BOUND;
		for (i = clip.top+1; i < clip.bottom; i++)
		{
			segment[0].y =
			segment[1].y = i*scaleHeight;
			Polyline(hdc, segment, 2);
		}

		/*	Every positive-slope line intersects the right or top border,
			and every negative-slope line intersects the left or top
			border.  So if we draw a positive-slope guide line through
			each point around the right and top borders, we draw all the
			visible lines.  Similarly, drawing a line through each point
			around the left and top borders covers the negative-slope
			lines.
		*/
		for (i = rect.top; i < rect.bottom; i += 2)
		{
			// Draw a positive-slope line through a point around the left border.
			drawGuide(hdc, +scaleWidth, scaleHeight,
				rect.left  * scaleWidth, i * scaleHeight);
			// Draw a negative-slope line through a point around the right border.
			drawGuide(hdc, -scaleWidth, scaleHeight,
				rect.right * scaleWidth, i * scaleHeight);
		}

		for (i = rect.left; i < rect.right; i += 2)
		{
			// Draw lines through a point around the top border.
			drawGuide(hdc, +scaleWidth, scaleHeight,
				i * scaleWidth, rect.top * scaleHeight);
			drawGuide(hdc, -scaleWidth, scaleHeight,
				i * scaleWidth, rect.top * scaleHeight);
		}

		DeleteObject(SelectObject(hdc, pen));
	}

	// Clip the paint region to the bounds of the shape.
	if (clip.left < 0)
		clip.left = 0;
	if (shape->width < clip.right)
		clip.right = shape->width;
	if (clip.top < 0)
		clip.top = 0;
	if (shape->height < clip.bottom)
		clip.bottom = shape->height;

	/*	It is permissible for the left bound to exceed the right
		bound or the top to exceed the bottom.  This occurs when
		the displayed shape does not fill the display, and we
		are asked to paint only the margin.  Then there is no
		work for us to do.
	*/

	// Select pen to use for outlining triangles interior to pieces.
	pen =(HPEN) SelectObject(hdc, GetStockObject(
		info->options.interiorPen ? BLACK_PEN : NULL_PEN));

	// Draw a filled polygon for each triangle in the shape in the update region.
	for (x = clip.left; x < clip.right; x++)
	for (y = clip.top; y < clip.bottom; y++)
		if (SPOT(shape, x, y) != EMPTY)
		{
			HBRUSH	brush;


			// Fill colors are indexed by piece IDs.
			brush = (HBRUSH) SelectObject(hdc, CreateSolidBrush(
				color[SPOT(shape, x, y)]));
			drawTriangle(hdc, x, y, parity, scaleWidth, scaleHeight);
			DeleteObject(SelectObject(hdc, brush));
		}
	
	SelectObject(hdc, pen);	// No delete since stock object was used.


	// Draw perimeters of pieces, if pen is not null.
	if (0 < info->perimeterPenThickness)
	{
		// Select pen to use for outlining perimeters of pieces.
		pen =(HPEN) SelectObject(hdc, CreatePen(PS_SOLID,
			info->perimeterPenThickness, info->options.perimeterPenColor));

		// Draw diagonal lines wherever the piece ID changes.
		for (y = clip.top; y < clip.bottom; y++)
		{
			// Compare EMPTY to first triangle (if it is in update region).
			for (x = clip.left; x == 0; x++)
				if (EMPTY               != SPOT(shape, x, y))
					drawDiagonal(hdc, x, y, parity, scaleWidth, scaleHeight);

			// Compare interior triangles.
			for (; x < clip.right; x++)
				if (SPOT(shape, x-1, y) != SPOT(shape, x, y))
					drawDiagonal(hdc, x, y, parity, scaleWidth, scaleHeight);

			// Compare last triangle to EMPTY (if it is in update region).
			for (x = shape->width; x <= clip.right; x++)
				if (SPOT(shape, x-1, y) != EMPTY            )
					drawDiagonal(hdc, x, y, parity, scaleWidth, scaleHeight);
		}

		/*	Draw horizontal borders on interior triangles.
		
			There is one horizontal side every two triangles as we
			move across a row, since one triangle is downward-pointing
			and the next is upward-pointing, so we increment x by
			2, having started at the appropriate offset (0 or 1)
			in each row.

			For each such side, if the triangles above and below it
			are for different pieces, we draw a line for the side.

			That describes the general case, which is in the middle
			set of for loops, below.  The top and bottom are
			variants of this.  In each of those cases, one of the
			triangles we would compare is outside the array.  It is
			actually an empty space.  Thus the first and third sets
			of loops below compare only a top or bottom spot to empty.

			The messy initial value for x is the x coordinate of the
			first even-parity triangle at the given coordinates or
			to its right.  (The even-parity triangles point downward
			and have coordinates x and y that sum to an even number.)
		*/
#define	EVENTRIANGLE(x, y)	((x) + ((x)+(y)+parity & 1))

		for (y = clip.top; y == 0; y++)
		for (x = EVENTRIANGLE(clip.left, y); x < clip.right; x += 2)
			if (EMPTY !=               SPOT(shape, x, y))
				drawHorizontal(hdc, x, y, scaleWidth, scaleHeight);

		for (; y < clip.bottom; y++)
		for (x = EVENTRIANGLE(clip.left, y); x < clip.right; x += 2)
			if (SPOT(shape, x, y-1) != SPOT(shape, x, y))
				drawHorizontal(hdc, x, y, scaleWidth, scaleHeight);

		for (y = shape->height; y <= clip.bottom; y++)
		for (x = EVENTRIANGLE(clip.left, y); x < clip.right; x += 2)
			if (SPOT(shape, x, y-1) != EMPTY            )
				drawHorizontal(hdc, x, y, scaleWidth, scaleHeight);

#undef	EVENTRIANGLE

		DeleteObject(SelectObject(hdc, pen));
	}

	return;
}


/*	Procedure for a window to display a shape.

	On creation, this window must be passed a pointer to a TriDisplayInfo
	structure.
*/
static LRESULT CALLBACK	TriDisplayWindow(
	HWND window, UINT message, WPARAM wParam, LPARAM lParam)
{
	/*	On creation, we are passed a pointer to space allocated for us.
		On subsequent calls, we use our saved copy of that pointer.
	*/
	TriDisplayInfo	*const info = message == WM_CREATE
		? (TriDisplayInfo *) ((LPCREATESTRUCT) lParam)->lpCreateParams
		: (TriDisplayInfo *) GetWindowLong(window, GWL_USERDATA);


	switch (message)
	{
		case WM_CREATE:
			// Save pointer passed to us at creation for future calls.
			SetWindowLong(window, GWL_USERDATA, (long) info);

			// Initialize data.
			info->cursor = LoadCursor((HINSTANCE) GetWindowLong(window, GWL_HINSTANCE),
				MAKEINTRESOURCE(IDC_EDIT));
			info->viewMode = VIEW_FIT;
			info->viewSize = 0;
			info->editMode = FALSE;
			info->shape = NULL;
			info->horz.scroll.cbSize = info->vert.scroll.cbSize =
				sizeof info->vert.scroll;
			info->horz.scroll.fMask = info->vert.scroll.fMask =
				SIF_RANGE | SIF_PAGE | SIF_POS;
			info->horz.velocity = info->vert.velocity = 0;
			info->horz.bar = SB_HORZ;
			info->vert.bar = SB_VERT;

			info->options.interiorPen = INTERIORPEN;
			info->options.perimeterPenColor = PERIMETERPENCOLOR;
			info->options.perimeterPenPixels = PERIMETERPENPIXELS;
			return 0;

		case WM_SETCURSOR:
			if (info->editMode)
			{
				SetCursor(info->cursor);
				return 0;
			}
			break;

		/*	When the user clicks in this window while we are in edit mode,
			make an edit to the shape.  When we are not in edit mode,
			notify the parent to exhibit this shape.
		*/
		case WM_LBUTTONDOWN:
			if (wParam == MK_LBUTTON && info->editMode)
			{
				SetCapture(window);
				info->captured = TRUE;
				invertTriangle(window, info, LOWORD(lParam), HIWORD(lParam), TRUE);
				return 0;
			}
			// When not in edit mode, fall through to exhibit our shape.
		case WM_NCLBUTTONDOWN:
			PostMessage(GetParent(window), WM_SETSHAPE,
				MAKELONG(GetWindowLong(window, GWL_ID), CE_IMPERATIVE),
				(long) info->shape);
			break;

		// As the user drags over our window, make edits to the shape.
		case WM_MOUSEMOVE:
			if (wParam == MK_LBUTTON && info->captured)
				invertTriangle(window, info, LOWORD(lParam), HIWORD(lParam), FALSE);
			return 0;

		case WM_LBUTTONUP:
			info->captured = FALSE;
			ReleaseCapture();
			return 0;

		case WM_MOVING:
		case WM_SIZING:
			// Disable moving and sizing by setting new location to current location.
			GetWindowRect(window, (RECT *) lParam);
			return 0;

		case WM_PAINT:
		{
			PAINTSTRUCT	ps;
			HDC			hdc;


			// If shape is volatile, redraw the whole thing.
			if (info->shape != NULL && info->shape->isVolatile)
				InvalidateRect(window, NULL, FALSE);

			hdc = BeginPaint(window, &ps);
			drawShape(hdc, info, ps.rcPaint);
			EndPaint(window, &ps);

			return 0;
		}

		case WM_SIZE:
			info->horz.length = LOWORD(lParam);
			info->vert.length = HIWORD(lParam);
			setScaling(window, info, FALSE);
			return 0;

		/*	Process a request to change the shape we display.  If
			CE_REDRAW is not set in HIWORD(wParam), the new shape has
			the same triangles as the old one, just with different
			piece IDs in them.  In that case, we do not need to
			recalculate transformations or scrolling and do not need
			to redraw the background.
		*/
		case WM_SETSHAPE:
		{
			Bool	redraw	= 0 != (HIWORD(wParam) & CE_REDRAW),
					newEdit	= 0 != (HIWORD(wParam) & CE_EDIT);


			if (newEdit != info->editMode)
			{
				/*	When changing the edit mode, we change the scaling
					and guide lines, so the shape must be redrawn.
				*/
				redraw = TRUE;

				// Note whether or not to suppress removal of scroll bars.
				info->horz.scroll.fMask = info->vert.scroll.fMask =
					SIF_RANGE | SIF_PAGE | SIF_POS |
						(newEdit ? SIF_DISABLENOSCROLL : 0);

				/*	When entering edit mode, we want to display scroll
					bars, even if they are disabled because the shape
					fits in the display, so that the scroll bars will
					not appear and disappear during editing.  That would
					cause client window size changes and induce changes
					in scroll position.  We can use SIF_DISABLENOSCROLL
					to prevent the scroll bars from disappearing, but
					there appears to be no way to force them to appear
					except to set a scroll range that requires it.  So
					do that here.
				*/
				if (newEdit)
				{
					const SCROLLINFO	force = 
						{ sizeof force, SIF_ALL, 0, 1, 0, 0, 0 };


					SetScrollInfo(window, SB_HORZ, &force, FALSE);
					SetScrollInfo(window, SB_VERT, &force, FALSE);
				}
				/*	When leaving edit mode, enable the scroll bars.
					Otherwise, they remain disabled even when the scroll
					range is set so that they are needed.
				*/
				else
					EnableScrollBar(window, SB_BOTH, ESB_ENABLE_BOTH);
			}

			/*	When fit-to-window scaling is set, we may have a cached size
				calculated when we entered edit mode.  When the shape changes,
				forget that size.
			*/
			if (redraw && info->viewMode == VIEW_FIT)
				info->viewSize = 0;
			info->editMode = newEdit;
			info->shape = (Shape *) lParam;
			setShape(window, info, redraw, FALSE);
			return 0;
		}
			
		case WM_GETSHAPE:
			* (Shape **) lParam = info->shape;
			return TRUE;

		// Process a request to change the view parameters.
		case WM_SETVIEW:
			/*	If zooming relative to previous view and not in fixed view,
				convert to fixed view.
			*/
			if (	(wParam == VIEW_ZOOMIN || wParam == VIEW_ZOOMOUT) &&
					info->viewMode != VIEW_FIXED)
			{
				info->viewMode = VIEW_FIXED;
				info->viewSize = (int) (2*info->horz.indexToPage *
					info->horz.extentDevice / COORDINATE_BOUND);
			}
			switch (wParam)
			{
				case VIEW_ZOOMIN:
					/*	Zoom in by multiplying view size by 3/2 --
						except integer 1 would not change, so set it to 2.
					*/
					info->viewSize = info->viewSize <= 1 ? 2 :
						info->viewSize * 3 / 2;
					break;

				case VIEW_ZOOMOUT:
					// Zoom out by multiplying view size by 2/3.
					info->viewSize = info->viewSize * 2 / 3;
					break;

				case VIEW_FIXED:
					info->viewMode = wParam;
					info->viewSize = (unsigned short) lParam;
					break;

				default:
				case VIEW_FIT:
					info->viewMode = wParam;
					info->viewSize = 0;
					break;
			}
			if (info->viewMode != VIEW_FIT)
				// Clip view size to meaningful range.
				info->viewSize = max(1, min(COORDINATE_BOUND, info->viewSize));

			setScaling(window, info, FALSE);
			return 0;

		// Process a request to change application options.
		case WM_SETOPTIONS:
			info->options = * (OptionsInfo *) lParam;
			setScaling(window, info, FALSE);
			return 0;
 
		// Convert arrow keys to scroll messages.
		case WM_KEYDOWN:
		{
			WORD	code;


			switch (wParam)
			{
				case VK_UP:		message = WM_VSCROLL; code = SB_LINEUP;		break;
				case VK_PRIOR:	message = WM_VSCROLL; code = SB_PAGEUP;		break;
				case VK_NEXT:	message = WM_VSCROLL; code = SB_PAGEDOWN;	break;
				case VK_DOWN:	message = WM_VSCROLL; code = SB_LINEDOWN;	break;
				case VK_HOME:	message = WM_VSCROLL; code = SB_TOP; 		break;
				case VK_END:	message = WM_VSCROLL; code = SB_BOTTOM; 	break;

				case VK_LEFT:	message = WM_HSCROLL; code = SB_LINELEFT;	break;
				case VK_RIGHT:	message = WM_HSCROLL; code = SB_LINERIGHT;	break;

				default:
					return 0;
			}
			wParam = MAKELONG(code, 0);
		}
		// Fall through into WM_HSCROLL and WM_VSCROLL.

		case WM_HSCROLL:
		case WM_VSCROLL:
		{
			ScrollInfo	*scrollInfo;
			int			desired;


			// Point to data for the dimension we are scrolling.
			scrollInfo = message == WM_HSCROLL ? &info->horz : &info->vert;

			/*	What's a good page increment?  A bit less than all
				of a page, so that some remains in the display
				to give the user visual continuity?  I find that
				important with text displays that jump.  However,
				the smooth scrolling in this program provides
				visual continuity, so scrolling an entire page at
				a time seems to be comfortable.
			*/

			// Start with current desired position, then modify it.
			desired = scrollInfo->scroll.nPos;
			switch (LOWORD(wParam))
			{
				case SB_LEFT:
					desired = scrollInfo->scroll.nMin;			break;
				case SB_PAGELEFT:
					desired = desired - scrollInfo->length;		break;
				case SB_LINELEFT:
					desired = desired - info->lineIncrement;	break;
				case SB_LINERIGHT:
					desired = desired + info->lineIncrement;	break;
				case SB_PAGERIGHT:
					desired = desired + scrollInfo->length;		break;
				case SB_RIGHT:
					desired = scrollInfo->posMax;				break;
				case SB_THUMBTRACK:
				case SB_THUMBPOSITION:
					desired = (short) HIWORD(wParam);			break;
				default:
					return 0;
			}
			
			// Clip the new desired position to the position bounds.
			if (desired < scrollInfo->scroll.nMin)
				desired = scrollInfo->scroll.nMin;
			else if (scrollInfo->posMax < desired)
				desired = scrollInfo->posMax;

			// Update the desired scroll position.
			scrollInfo->scroll.nPos = desired;
			SetScrollInfo(window, scrollInfo->bar, &scrollInfo->scroll, TRUE);

			// Set a timer to scroll display periodically until done.
			info->lastScroll = GetTickCount();
			SetTimer(window, 1, SCROLL_SPEED, NULL);

			/*	Fall through to do first increment.  This causes tracking
				to be fast, since we update on each tracking message.
			*/
		}

		/*	When we get a timer notification, scroll the window by
			a calculated amount to make scrolling smooth.
		*/
		case WM_TIMER:
		{
			RECT	rect;
			DWORD	n, thisScroll;


			/*	In case the SCROLL_SPEED setting is finer than the
				timer, apply multiple updates at each timer message.
			*/
			thisScroll = GetTickCount();
			// Calculate how many updates to apply.
			n = (thisScroll - info->lastScroll) / SCROLL_SPEED;
			/*	If an absurd amount of time has passed, do only
				100 updates.  This is enough to traverse 2,550 pixels,
				from standing start to standing stop.
			*/
			if (100 < n)
				n = 100;
			/*	Update time by amount allocated for the number of
				updates scheduled to have been executed, not by
				setting to the current time.  This tracks the
				target speed better by not losing fractions of
				intervals, as setting to the current time would do.
				Also, it does not include extra scroll messages
				that come in before the next scheduled update,
				since then n is zero but the smoothScroll routine
				still does at least one udpate.  This results in
				scrolling speeding up when the user is sending
				scroll commands.
			*/
			info->lastScroll += n * SCROLL_SPEED;

			GetClientRect(window, &rect);

			// Scroll window by calculated amounts.
			ScrollWindowEx(window,
				smoothScroll(window, info, &info->horz, n),
				smoothScroll(window, info, &info->vert, n),
				NULL, &rect, NULL, NULL, SW_INVALIDATE | SW_ERASE);

			// If we have reached both destinations, stop the timer.
			if (	info->horz.scroll.nPos == info->horz.scroll.nTrackPos &&
					info->vert.scroll.nPos == info->vert.scroll.nTrackPos)
				KillTimer(window, 1);

			return 0;
		}

		case WM_DESTROY:
			DestroyCursor(info->cursor);
			return 0;
	}

	return DefWindowProc(window, message, wParam, lParam);
}


/*	Register the TriDisplay window class.  This could be done in WinMain,
	but it's nice to collect all the code relevant to the window in one
	place.  For example, if we add code above that uses extra space in the
	window structure, we've got the code to tell Windows that right here.
	We return the identifier of the class.
*/
ATOM	TriDisplayRegister(HINSTANCE hInstance)
{
	static WNDCLASSEX	wcex = {
		sizeof wcex, 0, TriDisplayWindow, 0,
		0, 0, NULL, 0, (HBRUSH) (COLOR_WINDOW+1), NULL,
		"TriDisplay", NULL
	};

	ATOM	result;


	wcex.hInstance		= hInstance;
	wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);

	/*	We want a white background, not the system's window color,
		because many colors are used in the pieces and white and
		black are the only ones sure to be different.
	*/
	wcex.hbrBackground	= (HBRUSH) GetStockObject(WHITE_BRUSH);

	result = RegisterClassEx(&wcex);
	if (result == 0)
		DisplayWindowsError("TriSolve: Registering TriDisplay window class");

	//Initialize the array of colors.
	Sobol(NUM_COLORS, color);

	return result;
}


// Draw a shape on a print device context.
void	printShape(HDC hdc, const OptionsInfo *options, Shape *shape, RECT clip)
{
	TriDisplayInfo	info;
	

	memset(&info, 0, sizeof info);	// Lazy initialization, since largely unused.

	/*	Our job is easy, since the drawShape routine works for displays and
		printers.  Just set the context to the size we want the image to be,
		set the shape and options, and go.
	*/
	info.shape			= shape;
	info.horz.length	= GetDeviceCaps(hdc, HORZRES),
	info.vert.length	= GetDeviceCaps(hdc, VERTRES);
	info.options		= *options;
	info.viewMode		= VIEW_FIT;

	setScaling(NULL, &info, FALSE);

	drawShape(hdc, &info, clip);
}