/*	The code in this module originates in Numerical Recipes.
	The modifications are copyright 2002 by Eric Postpischil,
	http://edp.org.  See license information in index.html.
*/


#include	"Sobol.h"


#define BITS		8
#define DIMENSIONS	3


/*	This routine implements the Sobol sequence for BITS bits and DIMENSIONS
	dimensions.  The Sobol sequence is designed to give quasi-random numbers
	in a multi-dimensional space that are maximally separated.  I use it to
	generate color values to provide colors that are maximally different,
	to give good contrast between adjacent pieces in the display.

	The number of bits specifies the length of the sequence before it
	terminates or repeats, not necessarily the number of bits returned.

	The code here is adapted from Numerical Recipes in C by Press, Teukolsky,
	Vettering, and Flannery, Cambridge University Press, Second Edition
	(1996 printing), section 7.7, pages 312-313.  Comments here apply to
	the mechanics of the code -- See the text for information about its
	rationale.
*/
void	Sobol(int num, COLORREF *color)
{
	/*	Here are primitive polynomials modulo 2.  degree contains the
		degree of each polynomial, and polynomial contains the coefficients,
		encoded as bit i for the coefficient of x^(i+1).  x^0 and x^degree
		necessarily have coefficients of 1 and are not encoded.
	*/
	static const unsigned int	degree[DIMENSIONS] = { 1, 2, 3 },
								polynomial[DIMENSIONS] = { 0, 1, 1 };

	/* direction provides initial direction numbers for the Sobol
	   sequence.  Initial values are used in those elements
	   direction[j][k] where j < degree[k].  Each of these elements may
	   be an arbitrary odd number less than 2^(j+1).  The other elements
	   are calculated by the code; their initial values are ignored.
	*/
	static unsigned long	direction[BITS][DIMENSIONS] = {
		{ 1,  1,  1, },
		{ 0,  1,  3, }
	};

	/* n and x record the state of the current sequence.  n is the
	   current element number and x is the current point in the
	   multi-dimensional space. */
	unsigned long	x[DIMENSIONS];
	int				n;

	unsigned int	j;
	int				k, l;
	unsigned long	t, poly;
 

	/*	If vector data was initialized already, we are done.
		Otherwise, we initialize here.  This could be done at
		compile time.
	*/
	if (direction[0][0] == 1)
	{
		for (k = 0; k < DIMENSIONS; k++)
		{

			for (j = 0; j < degree[k]; j++)
				direction[j][k] <<= BITS-j-1;

			for (j = degree[k]; j < BITS; j++)
			{
				poly = polynomial[k];
				t = direction[ j-degree[k] ][ k ];
				t ^= t >> degree[k];
				for (l = degree[k]-1; l >= 1; l--)
				{
					if (poly & 1)
						t ^= direction[j-l][k];
					poly >>= 1;
				}
				direction[j][k] = t;
			}

		}
	}

	// Initialize for a new sequence.
	for (k = 0; k < DIMENSIONS; k++)
		x[k] = 0;
	// Starting with zero (black) makes the goal color conveniently gray.

	/* Fill in the array with the sequence. */
	for (n = 0; n < num; n++)
	{
		color[n] = RGB(x[0], x[1], x[2]);

		/* Find rightmost zero bit of n, which gives the index of the
		   vector to XOR to advance the Sobol sequence. */
		for (j = 0, t = n; j < BITS-1; j++, t >>= 1)
			if (! (t & 1)) break;
		/*	If (t & 1) here, there are no zero bits in n, and we are
			going to repeat values.  That's okay in this coloring
			application, but it would be an error in an application
			like random sampling that requires each sample to be
			distinct.
		*/

		/* XOR vector j into the current vector. */
		for (k = 0; k < DIMENSIONS; k++)
			x[k] ^= direction[j][k];
	}

	return;
}
