/*	Copyright 2002 by Eric Postpischil, http://edp.org.
	See license information in index.html.
*/


#include	"StdAfx.h"

#include	<ctype.h>
#include	<limits.h>
#include	<malloc.h>
#include	<stdio.h>
#include	<stdlib.h>


/*	Shapes and pieces are recorded primarily in two structures,
	called Shape and Piece.

	A shape is used for display, and it contains an image map for
	this purpose.  The image map is a two-dimensional array of
	piece identifiers.  The rows and columns of the array
	represent rows and columns of equilateral triangles.  The
	piece identifiers index information about the pieces, currently
	the color.  Special piece identifiers also mark special triangles,
	such as an empty space or a goal space.

	A shape is also used as a goal and work area, as the piece
	identifiers in its map record which spaces are empty, which are
	goals to be filled in, and which are filled with pieces.

	The shape structure contains other members:

		width and height give a bounding box for the shape; all
		of its points have x coordinates within [0, width),
		and all of its points have y coordinates within
		[0, height).

		numPoints records the number of points in the shape.

		parity is a bit indicating which way the triangle at
		(0, 0) in the shape's map is oriented.  If parity is
		even, the triangle is said to be downward-pointing,
		where the positive-x direction is right and the
		positive-y direction is down.  Specifically, the
		triangle at (0, 0) in the index-space coordinates
		(see coordinate systems in TriDisplay.cpp) of an
		even-parity shape has its vertices at Cartesian-space
		coordinates (0, 0), (2, 0), and (1, sqrt(3)/2).  For
		an odd-parity shape, the triangle would be pointing
		upward, with vertices at (0, sqrt(3)/2), (2, sqrt(3)/2),
		and (1, 0).

		Finally, the isVolatile structure member is a Boolean
		indicating that the shape is being worked on by another
		thread and may change asynchronously.  This is used to
		mark the Solver's work area so that the display routines
		know to redraw the entire shape if any part is redrawn.
		It may still change while being redrawn, but the drawing
		routines will at least use one color for each triangle,
		which is attractive.  If only an update region were
		repainted, a variety of line segments would be painted
		unattractively in different colors.  (The perimeter
		lines may also be drawn incorrectly, but this is not
		too distracting, as long as the user understands the
		image is of a work in progress.)

	A piece is used for geometric manipulation.  It uses a list of
	triangle coordinates instead of a map, so it may be easily
	rotated or used to index the map of a shape into which it is
	being placed.  This is of use in the Solver and in assembling
	pieces into an aggregate shape for display.

	The width, height, numPoints, and parity members of a piece
	serve the same purpose as in a shape.

	The points member is a one-dimensional array of Point
	structures, each giving the index-space coordinates of a
	triangle in the piece.

	The ID member is a piece identifier, with the same purpose
	as the piece identifiers in a shape map.
*/


#define	DIM(array)		(sizeof (array)/sizeof *(array))


/*	Coordinates, sizes, and numbers of points must be within
	fixed bounds.  Check.  Return TRUE iff number is good.
	Display error message if it is not.
*/
static Bool	checkBound(int n)
{
	if (n < SHRT_MIN || SHRT_MAX < n)
	{
		DisplayError("TriSolve: Bounds check", "Shape is too large.");
		return FALSE;
	}

	return TRUE;
}


/*	Rotate a point 120 degrees around the center of the triangle
	at (0, 0).  Return TRUE if successful.

	The coordinates must be in a frame with even parity.  That is,
	the triangle at (0, 0) must point in the positive-y direction.
*/
static Bool	rotate120(Point *out, Point in)
{
	int	x, y;


	/*	To rotate a point given its row and column numbers, we map
		those numbers to Cartesian coordinates, do a rigid geometric
		rotation on those coordinates, and map back.  The algebra here
		is:

			Given row and column numbers (x, y), the Cartesian
			coordinates of the center of the triangle (relative to
			the center of the triangle at (0, 0)) are
			(3*x, (3*y+f)*sqrt(3)), where f is 0 or 1 according
			to whether x+y is even or odd.  Let c and d be such
			that (c, d*sqrt(3)) = (3*x, (3*y+f)*sqrt(3)).

			We rotate (c, d*sqrt(3)) with a matrix multiplication:

				[     c     ]   [   -1/2      -sqrt(3)/2 ]
				[ d*sqrt(3) ]   [ sqrt(3)/2      -1/2    ]
		
			and the result is ( (-c-3*d)/2, (c-d)/2 * sqrt(3) ).

			Substituting for c and d in that result, we have
			( (-3*x-9*y-3*f)/2, (3*x-3*y-f)/2 * sqrt(3) ).

			To convert back from Cartesian coordinates of a
			triangle center to row and column numbers, we divide
			the abscissa by 3 and the ordinate by 3*sqrt(3),
			rounding down any fractions.  Rounding down the
			fraction is the inverse step of adding in f.

			The abscissa is a multiple of 3, so the divison
			yields (-x-3*y-f)/2.  If x+y is even, then so is
			-x-3*y, and f is 0, so the quotient equals (-x-3*y)/2.
			If x+y is odd, then f is 1, and the quotient
			equals floor((-x-3*y)/2).  That is, the quotient is
			-x-3*y divided by 2, rounded toward negative infinity
			(not zero).  The right-shift operator in C does this,
			so the abscissa we want may be computed as
			-x-3*y >> 1.

			Dividing the ordinate by 3*sqrt(3) yields
			(x-y)/2 - f/6.  If x+y is even, then so is x-y, and
			f is then 0, and there is no fraction.  If x+y is odd,
			then f is 1, and the result we want is
			floor((x-y-1/3)/2).  Since x-y is odd, we have
			floor((x-y)/2) = (x-y-1)/2 = floor((x-y-1/3)/2).
			Thus computing (x-y)/2 is correct if the division
			truncates toward negative infinity, not zero.  The
			right-shift operator in C does this.
	*/
	y = (in.x - in.y) >> 1;	// The ordinate is (x-y)/2.
	x = y - in.x - in.y;	// This equals (-x-3*y)/2.

	/*	We do not need to check y because the above arithmetic
		cannot put it out of bounds.
	*/
	if (!checkBound(x))
		return FALSE;

	out->x = x;
	out->y = y;

	return TRUE;
}


// Rotate a point 180 degrees around the point (1/2, 0).  Return TRUE if successful.
static Bool	rotate180(Point *out, Point in)
{
	if (!checkBound(1-in.x) || !checkBound(-in.y))
		return FALSE;

	/*	Rotating a point (x, y) 180 degrees around the origin
		yields (-x, -y), but this would give a triangle pointing
		down where we want a triangle pointing up.  Instead we
		rotate around (1/2, 0), which maps (x, y) to (1-x, -y).
	*/
	out->x = 1-in.x;
	out->y = -in.y;

	return TRUE;
}


// Reflect a point around the line x=0.  Return TRUE if succesful.
static Bool	reflect(Point *out, Point in)
{
	if (!checkBound(-in.x))
		return FALSE;

	out->x = -in.x;
	out->y = in.y;

	return TRUE;
}


// Allocate shape and fill in essential fields.
static Shape	*allocShape(void)
{
	Shape	*result;


	result = (Shape *) malloc(sizeof *result);
	if (result == NULL)
	{
		DisplayCError("TriSolve: Allocating shape");
		return NULL;
	}

	result->map = NULL;
	result->isVolatile = FALSE;

	return result;
}


// Allocate piece and fill in essential fields.
static Piece	*allocPiece(PieceID ID)
{
	Piece	*result;


	result = (Piece *) malloc(sizeof *result);
	if (result == NULL)
	{
		DisplayCError("TriSolve: Allocating piece");
		return NULL;
	}

	result->points = NULL;
	result->ID = ID;

	return result;
}


void	freeShape(Shape *shape)
{
	if (shape == NULL)
		return;

	free(shape->map);
	free(shape);
}


static void	freePiece(Piece *piece)
{
	if (piece == NULL)
		return;

	free(piece->points);
	free(piece);
}


// Copy a shape.
Shape	*copyShape(const Shape *shape)
{
	static const char	title[] = "TriSolve: Copying shape";

	Shape	*result;


	if (shape == NULL)
		return NULL;

	result = allocShape();
	if (result == NULL)
		return NULL;

	/*	Most of the scalars just copy, but isVolatile doesn't.
		The Solver's work area is volatile, but the copies it
		spawns off remain unchanged and are not volatile.
	*/
	*result = *shape;
	result->isVolatile = FALSE;

	result->map = (PieceID *) malloc(
		shape->width * shape->height * sizeof *result->map);
	if (result->map == NULL)
	{
		DisplayCError(title);
		free(result);
		return NULL;
	}
	memcpy(result->map, shape->map,
		shape->width * shape->height * sizeof *result->map);

	return result;
}


void	freeShapeList(ShapeList *shapes)
{
	// Traverse list.
	while (shapes != NULL)
	{
		ShapeList	*t;


		// Remember link before freeing list element.
		t = shapes->next;
		freeShape(shapes->shape);
		free(shapes);
		shapes = t;
	}
}


static void	freePieceList(PieceList *list)
{
	// Traverse list.
	while (list != NULL)
	{
		PieceList	*t;


		// Remember link before freeing list element.
		t = list->next;
		freePiece(list->piece);
		free(list);
		list = t;
	}
}


void	freePieces(Pieces *pieces)
{
	// Traverse list.
	while (pieces != NULL)
	{
		Pieces	*t;


		// Remember link before freeing list element.
		t = pieces->next;
		freePieceList(pieces->list);
		free(pieces);
		pieces = t;
	}
}


/*	This function defines the order we want points listed in.
	It is the standard comparison for sort, returning -1, 0, or 1
	to indicate less than, equal to, or greater than.
	We want points with the least x coordinate listed first, with
	ties broken by the y coordinate.

	This order must be the same as assumed in the Solver, so that
	it is easily able to locate the first point in this order, but
	it is not needed outside the Solver.  The normalize routine
	makes use of this order to identify the minimum and maximum x
	coordinates easily, but that can be readily changed.
*/
static int	comparePoints(Point *a, Point *b)
{
	if (a->x != b->x)
		return a->x < b->x ? -1 : 1;
	else if (a->y != b->y)
		return a->y < b->y ? -1 : 1;
	return 0;
}


// Create an image map of the points in a piece.
static Shape	*mapPoints(Piece *piece)
{
	Shape	*shape;
	int		i;


	if (piece == NULL)
		return NULL;

	shape = allocShape();
	if (shape == NULL)
		return NULL;

	shape->numPoints	= piece->numPoints;
	shape->width		= piece->width;
	shape->height		= piece->height;
	shape->parity		= piece->parity;

	shape->map = (PieceID *) calloc(
		shape->width * shape->height * sizeof *shape->map, 1);
	if (shape->map == NULL)
	{
		DisplayCError("TriSolve: Mapping points");
		freeShape(shape);
		return NULL;
	}

	/*	Map the points in a piece (given by coordinates) to points
		in a shape (laid out in a two-dimensional array).
	*/
	for (i = 0; i < shape->numPoints; i++)
		SPOT(shape, piece->points[i].x, piece->points[i].y) = piece->ID;

	return shape;
}


/*	Normalize the points in a piece.
	The points are sorted and translated to make the minimum
	x and y coordinates each 0.  The parity is adjusted by
	the translation.  The piece extents are set.  TRUE is
	returned if the operation is successful.
*/
static Bool	normalize(Piece *piece)
{
	int	i, minX, maxX, minY, maxY;


	if (piece->numPoints == 0)
	{
		piece->width = 0;
		piece->height = 0;
		return TRUE;
	}

	/*	Sort the points.  It's a waste to use qsort, since the list is
		probably tiny, and qsort has to handle arbitrary element sizes, but
		this is a small amount of processing that occurs only when reading
		shapes, so there's no point in writing other code.
	*/
	qsort(piece->points, piece->numPoints, sizeof *piece->points,
		(int (*)(const void *, const void *)) comparePoints);

	// Record extreme x coordinates.
	minX = piece->points[0].x;
	maxX = piece->points[piece->numPoints-1].x;

	// Find extreme y coordinates.
	minY = maxY = piece->points[0].y;
	for (i = 1; i < piece->numPoints; i++)
		if (piece->points[i].y < minY)
			minY = piece->points[i].y;
		else if (maxY < piece->points[i].y)
			maxY = piece->points[i].y;

	// Test whether translation will put coordinates out of bounds.
	if (!checkBound(maxX - minX + 1) || !checkBound(maxY - minY + 1))
		return FALSE;

	// Translate coordinates.
	for (i = 0; i < piece->numPoints; i++)
	{
		piece->points[i].x -= minX;
		piece->points[i].y -= minY;
	}

	piece->width = maxX - minX + 1;
	piece->height = maxY - minY + 1;
	piece->parity ^= minX+minY & 1;

	return TRUE;
}


// Transform a piece using a given transformation function
static Piece	*transform(Piece *piece, Bool (*function)(Point *, Point))
{
	int		i;
	Piece	*result;


	result = allocPiece(piece->ID);
	if (result == NULL)
		return NULL;

	result->points = (Point *) malloc(piece->numPoints * sizeof *result->points);
	if (result->points == NULL)
	{
		DisplayCError("TriSolve: Transforming piece");
		freePiece(result);
		return NULL;
	}

	result->numPoints = piece->numPoints;
	result->parity = 0;

	// Transform each point in piece.
	for (i = 0; i < piece->numPoints; i++)
	{
		Point	in;


		// The transformation functions require points in an even translation.
		in.x = piece->points[i].x - piece->parity;
		in.y = piece->points[i].y;
		if (! (*function)(&result->points[i], in))
		{
			freePiece(result);
			return NULL;
		}
	}

	if (!normalize(result))
	{
		freePiece(result);
		return NULL;
	}

	return result;
}


// Compare two pieces.  Return 0 if they are equal and 1 otherwise.
static int	comparePiece(Piece *a, Piece *b)
{
	int	i;


	if (a->numPoints != b->numPoints || a->parity != b->parity)
		return 1;
	for (i = 0; i < a->numPoints; i++)
		if (	a->points[i].x != b->points[i].x ||
				a->points[i].y != b->points[i].y)
			return 1;
	return 0;
}


// Determine whether a piece is in a list.
static Bool	in(Piece *piece, PieceList *list)
{
	PieceList	*p;


	for (p = list; p != NULL; p = p->next)
		if (comparePiece(piece, p->piece) == 0)
			return TRUE;
	return FALSE;
}


/*	This routine adds to a list all transformations of a piece that are
	not already in the list.  All transformations means all compositions
	of the functions listed just below.  The piece itself must not be in
	the list yet.  Calling this with an empty list yields a list of all
	transformations of the piece.

	If an error occurs, the input piece and list are freed and NULL is
	returned.
*/
static PieceList	*searchTransforms(Piece *piece, PieceList *list)
{
	static Bool	(* const function[])(Point *, Point) =
		{ rotate120, rotate180, reflect };

	PieceList	*result;
	Piece		*testPiece;
	int			i;


	/*	Start by putting the piece into the list, since it is given that
		the piece is not in the list.  Note that this is the mechanism
		by which transformations are added to the list, as finding a
		new transformation (below) results in a recursive call to this
		routine with the new piece.
	*/
	result = (PieceList *) malloc(sizeof *result);
	if (result == NULL)
	{
		DisplayCError("TriSolve: Searching piece transformations");
		freePieceList(list);
		freePiece(piece);
		return NULL;
	}
	result->next = list;
	result->piece = piece;

	/*	This is a dumb search with no knowledge of the transformation
		properties.  It tries every transformation on a piece and,
		when it finds a new piece, tries every transformation on it.
		This is obviously redundant as, for example, two reflections
		is the identity, as is three 120-degree rotations, and so are
		various combinations.  However, this is executed only when
		preparing pieces, not repeatedly during solving or other
		operations, so there is little value in optimizing it.  Also,
		leaving it ignorant of the transformations would allow it to
		be used with new transformations in the future.
	*/
	for (i = 0; i < DIM(function); i++)
	{
		// Try each transformation.
		testPiece = transform(piece, function[i]);
		if (testPiece == NULL)
		{
			freePieceList(result);
			return NULL;
		}
		
		if (! in(testPiece, result))
		{
			/*	If the transformation result is not in the list,
				add it to the list, along with all transformations
				of it that are not in the list yet.
			*/
			result = searchTransforms(testPiece, result);
			if (result == NULL)
				return NULL;
		}
		else
			/* If the transformation result is in the list, discard it. */
			freePiece(testPiece);
	}

	return result;
}


/*	Read a piece from the file and assign it the specified ID.
	The current position in the file is taken to be of even
	parity.  NULL return indicates EOF or error.
*/
Piece	*readPiece(FILE *file, PieceID ID)
{
	static const char	title[] = "TriSolve: Reading shape";

	Piece	*piece;
	Point	*points;
	int		allocated, i, j;
	char	c;


	piece = allocPiece(ID);
	if (piece == NULL)
		return NULL;

	/*	We scan through the input character-by-character,
		looking for points.  A piece is stored as a simple
		image map formed of spaces (for empty space) and
		non-space printable characters (for points).
		Newline characters of course indicate newlines,
		and other nonprintable characters indicate something's
		not right.

		As we scan, we keep track of the current row and
		column number.  When a point is found, its coordinates
		are appended to the list.
	*/
	piece->numPoints = allocated = 0;
	piece->parity = 0;
	i = j = 0;
	while (EOF != (c = getc(file)))
	{
		// On space, advance to next column.
		if (c == ' ')
			i++;
		// On newline, advance row and reset column.
		else if (c == '\n')
			j++, i = 0;
		// On printable non-space character, record triangle.
		else if (isprint(c) && !isspace(c))
		{
			if (!checkBound(piece->numPoints+1) || !checkBound(i) || !checkBound(j))
			{
				freePiece(piece);
				return NULL;
			}

			// If allocated list is full, allocate more.
			if (allocated <= piece->numPoints)
			{
				allocated = allocated == 0 ? 8 : 2*allocated;
				points = (Point *) realloc(piece->points,
					allocated * sizeof *piece->points);
				if (points == NULL)
				{
					DisplayCError(title);
					freePiece(piece);
					return NULL;
				}
				piece->points = points;
			}

			// Append point to list, and advance to next column.
			piece->points[piece->numPoints].x = i++;
			piece->points[piece->numPoints].y = j;
			piece->numPoints++;
		}
		// Anything else is noise in the file.
		else
		{
			DisplayError(title, "Unexpected character in shape file.");
			freePiece(piece);
			return NULL;
		}
	}

	if (ferror(file))
	{
		DisplayCError(title);
		freePiece(piece);
		return NULL;
	}

	// Release extra space.
	piece->points = (Point *) realloc(piece->points,
		piece->numPoints * sizeof *piece->points);

	if (!normalize(piece))
	{
		freePiece(piece);
		return NULL;
	}

	return piece;
}


/*	Read a shape from the file.
	NULL return indicates error.
*/
Shape	*readShape(FILE *file)
{
	Piece	*piece;
	Shape	*shape;


	// We get a shape by reading a piece and mapping its points.
	piece = readPiece(file, GOAL);
	shape = mapPoints(piece);
	freePiece(piece);
	return shape;
}


// Write a shape.  Return TRUE on success.
Bool	writeShape(FILE *file, const Shape *shape)
{
	const int	parity = shape->parity;

	int	i, j;


	// The shape is written as a simple character map.
	for (j = 0; j < shape->height; j++)
	{
		// Maintain parity with empty first column.
		if (parity)
			if (EOF == fputc(' ', file))
				break;
		for (i = 0; i < shape->width; i++)
			if (EOF == fputc(SPOT(shape, i, j) == EMPTY ? ' ' :
					i+j+parity & 1 ? 'A' : 'V', file))
				break;
		if (EOF == fputc('\n', file))
			break;
	}
	if (ferror(file))
	{
		DisplayCError("TriSolve: Writing shape");
		return FALSE;
	}
	return TRUE;
}


/*	Return TRUE if specified location in shape is empty.  The
	location may be outside the shape's map (which is of course
	empty).
*/
static Bool	isEmpty(Shape *shape, int x, int y)
{
	if (	x < 0 || shape->width <= x ||
			y < 0 || shape->height <= y)
		return TRUE;

	return SPOT(shape, x, y) == EMPTY;
}


/*	Test to see if a piece fits into an area at a specified
	location.  If it does, place it there and return TRUE.

	The piece must fit with no triangles of another piece
	adjacent to it -- we are looking for a fit for rearranging
	pieces, where we want the separation to be visually
	apparent to the user.  (The Solver uses a different fit
	routine, trying to fit pieces into the goal shape with
	no empty space.)
*/
static Bool	fits(Shape *area, Piece *piece, int x, int y)
{
	int	i;


	// The entire piece must fit into the area.
	if (	area->width < x + piece->width ||
			area->height < y + piece->height)
		return FALSE;

	// Examine where each triangle of the piece will go.
	for (i = 0; i < piece->numPoints; i++)
	{
		int	p, xt, yt;


		// Locate target spot.
		xt = x + piece->points[i].x;
		yt = y + piece->points[i].y;

		// Figure whether vertical neighbor is below or above.
		p = xt+yt+area->parity & 1 ? +1 : -1;

		// Check the destination spot and its neighbors.
		if (!isEmpty(area, xt  , yt  ))
			return FALSE;
		if (!isEmpty(area, xt-1, yt  ))
			return FALSE;
		if (!isEmpty(area, xt+1, yt  ))
			return FALSE;
#if 1	// Check only triangles with neighboring sides.
		if (!isEmpty(area, xt  , yt+p))
			return FALSE;
#else	// Check triangles with neighboring sides or vertices.
		if (!isEmpty(area, xt-2, yt  ))
			return FALSE;
		if (!isEmpty(area, xt+2, yt  ))
			return FALSE;
		if (!isEmpty(area, xt  , yt-1))
			return FALSE;
		if (!isEmpty(area, xt  , yt+1))
			return FALSE;
		if (!isEmpty(area, xt-1, yt-1))
			return FALSE;
		if (!isEmpty(area, xt-1, yt+1))
			return FALSE;
		if (!isEmpty(area, xt+1, yt-1))
			return FALSE;
		if (!isEmpty(area, xt+1, yt+1))
			return FALSE;
		if (!isEmpty(area, xt-2, yt+p))
			return FALSE;
		if (!isEmpty(area, xt+2, yt+p))
			return FALSE;
#endif
	}

	// Copy piece into area.
	for (i = 0; i < piece->numPoints; i++)
		SPOT(area, x + piece->points[i].x, y + piece->points[i].y) =
			piece->ID;
	area->numPoints += piece->numPoints;

	return TRUE;
}


/*	Try to fit a piece into an area.  If a fit is found,
	put the piece in place and return TRUE.
*/
static Bool	findFit(Shape *area, PieceList *piece)
{
	PieceList	*q;
	int			x, y;


	// Try each y coordinate for translation.
	for (y = 0; y < area->height; y++)
		// Try each orientation of the piece.
		for (q = piece; q != NULL; q = q->next)
			// Try each x coordinate that gives translation proper parity.
			for (x = y+area->parity+q->piece->parity & 1; x < area->width; x += 2)
				if (fits(area, q->piece, x, y))
					return TRUE;

	return FALSE;
}


/*	Expand a shape's map to make more room.  The map is expanded
	by one unit in the width or the height, whichever brings it
	close to the target width and height.  Return TRUE if successful.
*/
static Bool	expandMap(Shape *shape, int width, int height)
{
	PieceID	*map;
	int		x, y, newWidth, newHeight;


	// Start with current width and height.
	newWidth = shape->width;
	newHeight = shape->height;

	/*	Increase the dimension that is further from its target, allowing
		for aspect ratio.
	*/
	if (newWidth * height < newHeight * width * (HEIGHT/WIDTH))
	{
		if (!checkBound(++newWidth))
			return FALSE;
	}
	else
	{
		if (!checkBound(++newHeight))
			return FALSE;
	}

	// Get space for new map.
	map = (PieceID *) calloc(newWidth * newHeight * sizeof *map, 1);
	if (map == NULL)
		return FALSE;

	// Copy old map into new map.
	for (x = 0; x < shape->width; x++)
	for (y = 0; y < shape->height; y++)
		map[y * newWidth + x] = SPOT(shape, x, y);
	shape->width = newWidth;
	shape->height = newHeight;

	// Replace and free old map.
	free(shape->map);
	shape->map = map;

	return TRUE;
}


// Return TRUE if a row of a shape is empty.
static Bool	emptyRow(Shape *shape, int y)
{
	int	x;


	for (x = 0; x < shape->width; x++)
		if (SPOT(shape, x, y) != EMPTY)
			return FALSE;
	return TRUE;
}


// Return TRUE if a column of a shape is empty.
static Bool emptyColumn(Shape *shape, int x)
{
	int	y;


	for (y = 0; y < shape->height; y++)
		if (SPOT(shape, x, y) != EMPTY)
			return FALSE;
	return TRUE;
}


/*	Remove empty rows and columns at shape edges.
	Return TRUE if shape changes.
*/
Bool	compressShape(Shape *shape)
{
	int		x, y, originX, originY, width, height;


	if (shape == NULL)
		return FALSE;

	if (shape->numPoints == 0)
	{
		shape->width = 0;
		shape->height = 0;
		shape->parity = 0;
		free(shape->map);
		shape->map = NULL;
		return TRUE;
	}

	// Check left columns.
	for (originX = 0; emptyColumn(shape, originX); originX++)
		;

	// Check right columns.
	for (width = shape->width; emptyColumn(shape, width-1); width--)
		;

	// Check top rows.
	for (originY = 0; emptyRow(shape, originY); originY++)
		;

	// Check bottom rows.
	for (height = shape->height; emptyRow(shape, height-1); height--)
		;


	if (originX != 0 || originY != 0 || width != shape->width || height != shape->height)
	{
		/*	Since we are compressing the shape, the copy can be done
			in-place.
		*/
		for (y = originY; y < height; y++)
		for (x = originX; x < width; x++)
			shape->map[(y-originY)*(width-originX) + x-originX] = SPOT(shape, x, y);

		// Reduce allocated space.
		width -= originX;
		height -= originY;
		shape->map = (PieceID *) realloc(shape->map,
			width * height * sizeof *shape->map);
		shape->width = width;
		shape->height = height;
		shape->parity ^= originX+originY & 1;
		return TRUE;
	}

	return FALSE;
}


/*	Make one shape containing all the pieces, so they can be displayed
	together.  The width and height give a target ratio for the
	the resulting area.
	
	We will try putting each piece into a growing area.  This is an
	extraordinarily wasteful algorithm, as it is dumb and grows the
	area only one unit at a time, when several units might be required
	to get a fit.  But it will be executed rarely, only when arranging
	pieces for display.
*/
Shape	*aggregatePieces(Pieces *pieces, int width, int height)
{
	Pieces	*p;
	Shape	*result;


	if (pieces == NULL)
		return NULL;

	result = allocShape();
	if (result == NULL)
		return NULL;

	// Protect against degenerate dimensions.
	if (width <= 0 || height <= 0)
		width = height = 1;

	//Initialize shape dimensions.
	result->numPoints = 0;
	result->width = result->height = 0;
	result->parity = 0;

	// Place each piece.
	for (p = pieces; p != NULL; p = p->next)
		// While piece does not fit, expand area.
		while (!findFit(result, p->list))
			if (!expandMap(result, width, height))
			{
				// If expand fails, give up.
				freeShape(result);
				return NULL;
			}

	/*	We compact the shape here.  The first column might be
		empty due to parity, and a row or column could have been
		added that wasn't used, because it wasn't until a column
		or row was added that a piece fit.
	*/
	compressShape(result);

	return result;
}


/*	Here is a sample shape for display while changing options.
	It is completely static and should not be freed.
*/
Shape	*sampleShape(void)
{
	static PieceID	map[4][7] = {
		{ 0, 4, 4, 4, 3, 3, 0 },
		{ 2, 4, 2, 5, 5, 3, 3 },
		{ 2, 2, 2, 6, 5, 5, 5 },
		{ 0, 6, 6, 6, 6, 6, 0 } };
	static Shape	shape = { DIM(*map), DIM(map), 24, 0, FALSE, (PieceID *) map };


	return &shape;
}


// Return a shape allocated for the New command.
Shape	*newShape(void)
{
	static const Shape	shape = { 0, 0, 0, 0, FALSE, NULL };


	return copyShape(&shape);
}


/*	Check whether a point (location given by x and y) in a shape
	has the same identity as a piece currently being considered.
	If it does, append it to a list and remove it from the shape.
	The place to append it is pointed to by *append.
*/
static void	checkPart(Point **append, Shape *shape, int x, int y, PieceID identity)
{
	// Point is part of piece if it matches piece's identity.
	if (SPOT(shape, x, y) == identity)
	{
		// Append point to list.
		(*append)->x = x;
		(*append)->y = y;
		++*append;

		// Remove point from shape.
		SPOT(shape, x, y) = EMPTY;
		shape->numPoints--;
	}
}


/*	Extract or color the piece at the indicated location in
	the shape.  Contiguous triangles form part of the piece
	if they have the same ID as the triangle at the given
	location.

	If color is NULL, the piece is extracted.  The new piece
	is assigned the given ID, made into a list of its
	transformations, and inserted into the list passed in
	pieces.  The resulting list is returned.  If an error
	occurs, the list is not freed.

	If color is not NULL, the piece is colored.  The piece
	IDs are set in color, while shape is used as a work
	area from which the piece can be removed.  The input
	value of pieces is returned unchanged as the function
	result.
*/
static Pieces	*extractPiece(Pieces *pieces, Shape *shape,
	int x, int y, PieceID ID, Shape *color)
{
	const char	title[] = "TriSolve: Extracting piece";

	Pieces		*result;
	PieceList	*list;
	Piece		*piece;
	Point		*points, *pending, *end;
	PieceID		identity;


	/*	The points array is a list of points in the piece being
		extracted.  pending and end point to locations in the
		list:

			Points in [points, pending) have been examined
			for neighbors and are done.

			Points in [pending, end) have been found and
			added to the piece but have not been examined
			for neighbors.

			Newly found points are added at end.
	*/

	// Get enough space to hold all points in shape.
	points = (Point *) malloc(shape->numPoints * sizeof *points);
	if (points == NULL)
	{
		DisplayCError(title);
		if (color == NULL)
			freePieces(pieces);
		return NULL;
	}
	pending = end = points;

	// Get ID of first triangle.
	identity = SPOT(shape, x, y);

	// Put first triangle on list.
	checkPart(&end, shape, x, y, identity);

	/*	Continue seeking new neighbors while there are points
		whose neighbors have not been checked.
	*/
	while (pending < end)
	{
		x = pending->x;
		y = pending->y;
		pending++;

		// Check left neighbor.
		if (0 < x)
			checkPart(&end, shape, x-1, y, identity);

		// Check right neighbor.
		if (x < shape->width-1)
			checkPart(&end, shape, x+1, y, identity);

		/*	Check neighbor above or below, according to
			the triangle's orientation.
		*/
		if (x+y+shape->parity & 1)
		{
			if (y < shape->height-1)
				checkPart(&end, shape, x, y+1, identity);
		}
		else
		{
			if (0 < y)
				checkPart(&end, shape, x, y-1, identity);
		}
	}

	// If we are coloring and not extracting, do that now.
	if (color != NULL)
	{
		for (pending = points; pending < end; pending++)
			SPOT(color, pending->x, pending->y) = ID;
		free(points);
		return pieces;
	}
	// We are not coloring.  Continue to extract piece.

	// Put what we've found into a piece.
	piece = allocPiece(ID);
	if (piece == NULL)
	{
		freePieces(pieces);
		free(points);
		return NULL;
	}
	piece->numPoints = end - points;
	piece->parity = shape->parity;

	// Release space in list of points.
	piece->points = (Point *) realloc(points, piece->numPoints * sizeof *points);

	normalize(piece);

	// Create a list of transformations of the piece.
	list = searchTransforms(piece, NULL);
	if (list == NULL)
	{
		freePieces(pieces);
		return NULL;
	}
	
	// Insert this piece's transforms in the list of pieces.
	result = (Pieces *) malloc(sizeof *result);
	if (result == NULL)
	{
		DisplayCError(title);
		freePieceList(list);
		freePieces(pieces);
		return NULL;
	}
	result->next = pieces;
	result->list = list;
	result->used = FALSE;

	return result;
}


/*	Extract all pieces from or color all pieces in a shape.

	If extractP is TRUE, the pieces are colored in place.
	(Pieces *) TRUE is returned if any gray triangles were
	colored.  (Pieces *) FALSE is returned otherwise.

	If extractP is FALSE, the pieces are colored in place
	and a list of pieces is returned (or NULL on error).
*/
Pieces	*extractPieces(Shape *shape, Bool extractP)
{
	Shape	*color, *work;
	Pieces	*result;
	Bool	used[NUM_COLORS];
	PieceID	lastKnownUsed = GOAL;
	int		x, y;


	if (shape == NULL)
		return NULL;

	/*	Set parameter to be passed to extractPiece to select
		coloring or extracting, and set initial result.
	*/
	if (extractP)
	{
		color = NULL;
		result = NULL;
	}
	else
	{
		color = shape;
		result = (Pieces *) FALSE;
	}

	// Mark all colors unused.
	memset(used, FALSE, sizeof used);

	// Make copy to work in.
	work = copyShape(shape);
	if (work == NULL)
		return NULL;

	// Scan for colored pieces first, to preserve colors.
	for (x = 0; x < work->width; x++)
	for (y = 0; y < work->height; y++)
		if (PIECE <= SPOT(work, x, y))
		{
			used[SPOT(work, x, y)] = TRUE;
			result = extractPiece(result, work, x, y, SPOT(work, x, y), color);
			if (extractP && result == NULL)
			{
				freeShape(work);
				return NULL;
			}
		}

	// Scan for any pieces next, coloring as necessary.
	for (x = 0; x < work->width; x++)
	for (y = 0; y < work->height; y++)
		if (SPOT(work, x, y) != EMPTY)
		{
			while (used[++lastKnownUsed])
				;
			if (lastKnownUsed == 0)
			{
				DisplayError("TriSolve: Extracting pieces", "Too many pieces.");
				if (extractP)
					freePieces(result);
				freeShape(work);
				return NULL;
			}
			used[lastKnownUsed] = TRUE;
			result = extractPiece(result, work, x, y, lastKnownUsed, color);
			if (extractP && result == NULL)
			{
				freeShape(work);
				return NULL;
			}
			// If any piece is colored in color mode, return TRUE.
			else if (!extractP)
				result = (Pieces *) TRUE;
		}

	freeShape(work);
	return result;
}


/*	Toggle the presence of a triangle in a shape.  The triangle may
	be outside the current bounds of the shape.  The shape is expanded
	as necessary.  All size changes are symmetric, to help
	keep limit shape motion in the display.  It is not contracted, as
	that could force motion of the shape in the display as the scroll
	bounds might have to be contracted to stay within limits.
	
	If an error occurs, a message is displayed and the shape is
	returned unchanged.

	The amount of change in the piece's origin is returned as a vector
	to the old origin from the new origin, so the caller can adjust any
	coordinates they have relative to the old origin.
*/
Point	toggleTriangle(Shape *shape, int x, int y)
{
	const int	width = shape->width,
				height = shape->height;
	const Point	zero = { 0, 0 };

	Point	change;


	// Start with unchanged bounds.
	change = zero;

	/*	Check whether the new point extends the shape's bounds
		in any direction.  Record any changed bound.
	*/
	if (x < 0)
		change.x = -x;
	else if (width <= x)
		change.x = x+1 - width;
	if (y < 0)
		change.y = -y;
	else if (height <= y)
		change.y = y+1 - height;

	// If any of the checks so far succeeded, the shape is getting bigger.
	if (0 < change.x || 0 < change.y)
	{
		// Check to see if it would be too big.
		if (!checkBound(width + 2*change.x))
			return zero;
		if (!checkBound(height + 2*change.y))
			return zero;
	}
	/*	If the checks so far did not detect expansion,
		the shape is not getting bigger, and the subject
		point is within existing bounds.
	*/
	else
		// Triangle is within existing bounds.  Is it going away?
		if (SPOT(shape, x, y) != EMPTY)
		{
			// Remove triangle and we're done.
			SPOT(shape, x, y) = EMPTY;
			shape->numPoints--;
			return zero;
		}
		// Else there is a point to add in bounds.  We have to color it.

	if (!checkBound(shape->numPoints + 1))
		return zero;

	// If map dimensions changed, we need to recreate the map.
	if (change.x != 0 || change.y != 0)
	{
		PieceID	*map;
		RECT	intersect;
		int	i, j;


		// Allocate space for new map.
		map = (PieceID *) calloc(
			(width + 2*change.x) * (height + 2*change.y),
			sizeof *map);
		if (map == NULL)
		{
			DisplayCError("TriSolve: Editing shape");
			return zero;
		}

		// Copy intersecting portion of old map into new map.
		SetRect(&intersect, 0, 0, width, height);
		InflateRect(&intersect, min(0, change.x), min(0, change.y));
		for (i = intersect.left; i < intersect.right; i++)
		for (j = intersect.top; j < intersect.bottom; j++)
			map[(j+change.y) * (width+2*change.x) + (i+change.x)] =
				SPOT(shape, i, j);

		// Release old map and update shape members.
		free(shape->map);
		shape->map = map;
		shape->width += 2*change.x;
		shape->height += 2*change.y;
		shape->parity ^= change.x+change.y & 1;

		// Translate coordinates from old shape to new shape.
		x += change.x;
		y += change.y;
	}

	/*	If we are adding a triangle, it can be added now, after we allocated
		space for the new dimensions (if necessary) and copied the map.
	*/
	{
		PieceID	spot, left, right, vertical;
		int		p;


		/*	If spot has neighbors of only one ID, give it that ID.
			Otherwise, give it the GOAL ID.  The tests for this
			seem somewhat convoluted.  Is there a neater sequence?
		*/
		p = x+y+shape->parity & 1 ? +1 : -1;
		left		= x-1 < 0							? EMPTY : SPOT(shape, x-1, y);
		right		= shape->width <= x+1				? EMPTY : SPOT(shape, x+1, y);
		vertical	= y+p < 0 || shape->height <= y+p	? EMPTY : SPOT(shape, x, y+p);

		spot = left;

		if (spot == EMPTY)
			spot = right;
		else if (right != EMPTY && spot != right)
			spot = GOAL;

		if (spot == EMPTY)
			spot = vertical;
		else if (vertical != EMPTY && spot != vertical)
			spot = GOAL;

		if (spot == EMPTY)
			spot = GOAL;

		SPOT(shape, x, y) = spot;
		shape->numPoints++;
	}

	return change;
}