/*	Copyright 2002 by Eric Postpischil, http://edp.org.
	See license information in index.html.
*/


#include	"StdAfx.h"
#include	"resource.h"


static DWORD	tls = -1;	// thread-local storage index


typedef struct
{
		OptionsInfo	options;		// application options
		HDC			hdc;			// printer device context
		Shape		*shape;			// shape to print
		Bool		printToFile;	// user requests print-to-file
}	PrintParameters;		// parameters passed to print thread


typedef struct
{
		HWND	window;				// abort-dialog window
		Bool	abort;				// user requests abort
}	PrintInfo;				// internal data of print thread


// Message handler for print abort dialog.
BOOL CALLBACK	abortDialog(HWND dialog, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:
			return TRUE;

		// If user clicks button, set flag to abort printing.
		case WM_COMMAND:
		{
			/*	Since there is no context parameter, get our context 
				from thread-local storage.
			*/
			PrintInfo	* const info = (PrintInfo *) TlsGetValue(tls);


			info->abort = TRUE;		// We are aborting.
			info->window = NULL;	// Send no more messages to this window.
			DestroyWindow(dialog);	// Destroy window.
			return TRUE;
		}
	}
    return FALSE;
}


/*	This is the procedure called by the print operations to
	find out if printing should be aborted.  In spite of being
	called an abort procedure instead of a continue procedure,
	it returns TRUE iff printing should continue.
*/
BOOL CALLBACK	abortProc(HDC hdc, int error)
{
	// Since there is no context parameter, get our context from thread-local storage.
	const PrintInfo	* const info = (PrintInfo *) TlsGetValue(tls);

    MSG	message;

 
	/*	In order to know whether the user wants to abort printing,
		the abort dialog must operate.  That means we have to get
		messages for it and dispatch them.  So a message dispatch
		loop follows.

		Printing in a separate thread has another benefit aside
		from allow other operations to continue in parallel.  We
		get no messages for anything but the abort dialog.  So we
		do not need to translate accelerator keys or deal with
		the other baggage in the main thread's message dispatch loop.
	*/

    // Get message for this thread or allow other threads and processes to run.
 	while (PeekMessage(&message, NULL, 0, 0, PM_REMOVE))
	{
		// Dispatch messages for abort dialog window.
		if (info->window != NULL && IsDialogMessage(info->window, &message))
			;
		/*	Dispatch regular messages.  (As noted above, there will
			be no messages for other windows.  But we have no
			guarantee that a message for the dialog will not be sent
			in a way that is not handled by IsDialogMessage, so we
			dispatch here for safety.)
		*/
		else 
		{
			TranslateMessage(&message);
			DispatchMessage(&message);
		}
	}
 
    return !info->abort;
}

/*	This is the main routine for the print thread.  The user has
	been prompted for print job information, and we are passed
	everything neeed to specify the printing in a PrintParameters
	structure.

	The structure and the shape it points to must be freed by
	this routine, and the device context must be deleted.
*/
static void	printMain(PrintParameters *parameters)
{
	static const char	title[] = "TriSolve: Printing";

	static const DOCINFO	docInfoTemplate = {
		sizeof docInfoTemplate,
				// structure size
		"TriSolve Shape",
				// document name
		NULL,	// output file name
		NULL,	// data type
		0,		// banding flag
	};

	DOCINFO		docInfo = docInfoTemplate;
	PrintInfo	*info;
	int			result;
	RECT		clip;


	// Print in the background.
	SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_BELOW_NORMAL);

	/*	Windows won't pass context information to the abort procedure,
		so we have to do it via another mechanism.  Start by getting
		space to hold our data.
	*/
	info = (PrintInfo *) malloc(sizeof *info);
	if (info == NULL)
	{
		DisplayCError(title);
		DeleteDC(parameters->hdc);
		freeShape(parameters->shape);
		free(parameters);
		return;
	}


	/*	Now store the address of our data in thread-local storage.  Note
		that tls is one value for the entire process that indexes a
		different storage location in each thread.  Thus tls is a static
		variable.
	*/
	TlsSetValue(tls, info);

	// Presume printing is not being aborted and create abort dialog.
	info->abort = FALSE;
	info->window = CreateDialog(NULL, MAKEINTRESOURCE(IDD_ABORT), NULL, abortDialog);

	// Tell print routines how to check for abort request.
	SetAbortProc(parameters->hdc, abortProc);

	// Set output to file if user requested it.
	if (parameters->printToFile)
		docInfo.lpszOutput = "FILE:";

	/*	This is a simple print job:  start document, start page,
		print shape, end page, end document.  The only complication
		is to check for errors after each call.  If this sequence
		gets any longer, it may be better to write it as a 
		do {...} while (0) "loop" using break statements to exit if
		an error occurs.
	*/
	if (0 < (result = StartDoc(parameters->hdc, &docInfo)))
		if (0 < (result = StartPage(parameters->hdc)))
		{
			/*	It would be trivial to support banding here, but that is
				obsolete in Win32.  printShape prints the portion of the
				region we pass in clip, and it scales the image to fit in
				the entire region of the device.  That's perfect for
				banding.  However, if we want to print a complete shape in
				only part of the device region, we'll modify printShape to
				use the clip rectangle for both scaling and clipping.
			*/
			SetRect(&clip, 0, 0,
				GetDeviceCaps(parameters->hdc, HORZRES),
				GetDeviceCaps(parameters->hdc, VERTRES));
			printShape(parameters->hdc, &parameters->options, parameters->shape, clip);

			if (0 < (result = EndPage(parameters->hdc)))
				result = EndDoc(parameters->hdc);
		}

	/*	Display error, except abort requests, since the user already knows
		about them.  The old bit for indicating an error had already been
		reported to the user appears to be obsolete and nonfunctional.
	*/
	if (result <= 0 && result != SP_APPABORT && result != SP_USERABORT)
		DisplayWindowsError(title);

	// Free our local context.
	if (!info->abort)
		DestroyWindow(info->window);
	free(info);

	// Free parameters.
	DeleteDC(parameters->hdc);
	freeShape(parameters->shape);
	free(parameters);
}


#include	<commdlg.h>
#include	<process.h>


// Execute Print command.
void	print(HWND window, const OptionsInfo *options, const Shape *shape)
{
	static const char	title[] = "TriSolve: Initializing for printing";

	static const	PRINTDLG	printDlgTemplate = {
		sizeof printDlgTemplate,
				// structure size
		NULL,	// owner window
		NULL,	// device mode structure
		NULL,	// device names structure
		NULL,	// address for device context handle
		PD_NOPAGENUMS | PD_NOSELECTION |
		PD_RETURNDC | PD_USEDEVMODECOPIESANDCOLLATE,
				// flags
		0,		// starting page
		0,		// ending page
		1,		// minimum starting page
		1,		// maximum ending page
		1,		// number of copies
		NULL,	// instance for templates
		NULL,	// application data for hooks
		NULL,	// print hook
		NULL,	// setup hook
		NULL,	// print template name
		NULL,	// setup template name
		NULL,	// print template
		NULL,	// setup template
	};

	PRINTDLG		printDlg = printDlgTemplate;
	BOOL			result;
	unsigned long	thread;
	PrintParameters	*print;


	// Get space to pass print parameters to thread.
	print = (PrintParameters *) malloc(sizeof *print);
	if (print == NULL)
	{
		DisplayCError(title);
		return;
	}

	// Print shape with user's current options.
	print->options = *options;

	// Give print thread its own copy of shape, which it must free.
	print->shape = copyShape(shape);

	/*	We might have no shape to print, either because we were called
		with no shape or because copyShape failed.  (If so, it displayed
		an error message already.)

		In either case, if there is no shape, leave now.
	*/
	if (print->shape == NULL)
	{
		free(print);
		return;
	}

	/*	Initialize thread-local storage.  We need to allocate the storage in
		a central thread, after which each thread can use it for its own context.
	*/
	if (tls == -1)
	{
		tls = TlsAlloc();
		if (tls == -1)
		{
			DisplayWindowsError(title);
			freeShape(print->shape);
			free(print);
			return;
		}
	}

	// Prompt user about printing and get printer device context.
	result = PrintDlg(&printDlg);

	// Free device information we do not want.
	if (printDlg.hDevMode != NULL)
		GlobalFree(printDlg.hDevMode);
	if (printDlg.hDevNames != NULL)
		GlobalFree(printDlg.hDevNames);

	if (!result)
	{
		// If an error occurred (not including user cancel, 0), display message.
		if (CommDlgExtendedError() != 0)
			DisplayError("TriSolve: Creating Print dialog",
				"Common dialog box error.");
		freeShape(print->shape);
		free(print);
		return;
	}

	print->hdc = printDlg.hDC;
	print->printToFile = printDlg.Flags & PD_PRINTTOFILE ? TRUE : FALSE;

	// Start print thread.
	thread = _beginthread((void (*)(void *)) printMain, 0, print);
	if (thread == NO_THREAD)
	{
		DisplayWindowsError("TriSolve: Starting print thread");
		DeleteDC(printDlg.hDC);
		return;
	}
}